package com.atlassian.jira.event.user.anonymize;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;

import javax.annotation.ParametersAreNonnullByDefault;
import java.time.Duration;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * This event is sent when a handler related to user anonymization is being called, e.g. to perform the anonymization
 * operation or to get entities affected by the update.
 *
 * @since v8.6
 */
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
@EventName("jira.user.anonymization.handler.executed")
public class UserAnonymizationHandlerExecutedEvent {

    private enum Operation {AFFECTED_ENTITIES, UPDATE, NUMBER_OF_TASKS}

    private final Operation operation;
    private final String pluginKey;
    private final String handlerKey;
    private final boolean success;
    private final long duration;

    private UserAnonymizationHandlerExecutedEvent(Operation operation, String pluginKey, String handlerKey, boolean success, Duration duration) {
        this.operation = requireNonNull(operation);
        this.pluginKey = requireNonNull(pluginKey);
        this.handlerKey = requireNonNull(handlerKey);
        this.success = success;
        this.duration = requireNonNull(duration).toMillis();
    }

    public static UserAnonymizationHandlerExecutedEvent affectedEntities(String pluginKey, String handlerKey, boolean success, Duration duration) {
        return create(Operation.AFFECTED_ENTITIES, pluginKey, handlerKey, success, duration);
    }

    public static UserAnonymizationHandlerExecutedEvent update(String pluginKey, String handlerKey, boolean success, Duration duration) {
        return create(Operation.UPDATE, pluginKey, handlerKey, success, duration);
    }

    public static UserAnonymizationHandlerExecutedEvent numberOfTasks(String pluginKey, String handlerKey, boolean success, Duration duration) {
        return create(Operation.NUMBER_OF_TASKS, pluginKey, handlerKey, success, duration);
    }

    private static UserAnonymizationHandlerExecutedEvent create(Operation operation, String pluginKey, String handlerKey, boolean success, Duration duration) {
        return new UserAnonymizationHandlerExecutedEvent(operation, pluginKey, handlerKey, success, duration);
    }

    /**
     * @return the operation that was executed
     */
    public Operation getOperation() {
        return operation;
    }

    /**
     * @return the key of the plugin that contains the executed handler
     */
    public String getPluginKey() {
        return pluginKey;
    }

    /**
     * @return the key of the executed handler
     */
    public String getHandlerKey() {
        return handlerKey;
    }

    /**
     * @return whether the handler executed successfully or not
     */
    public boolean isSuccess() {
        return success;
    }

    /**
     * @return the time in milliseconds it took the handler to execute
     */
    public long getDuration() {
        return duration;
    }

    /**
     * Used only to make assertions in tests easier. The duration parameter was skipped as it is too unpredictable.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserAnonymizationHandlerExecutedEvent that = (UserAnonymizationHandlerExecutedEvent) o;
        return operation == that.operation &&
                success == that.success &&
                Objects.equals(pluginKey, that.pluginKey) &&
                Objects.equals(handlerKey, that.handlerKey);
    }

    /**
     * Used only to make assertions in tests easier. The duration parameter was skipped as it is too unpredictable.
     */
    @Override
    public int hashCode() {
        return Objects.hash(operation, success, pluginKey, handlerKey);
    }

    @Override
    public String toString() {
        return "UserAnonymizationHandlerExecutedEvent{" +
                "operation=" + operation +
                ", pluginKey='" + pluginKey + '\'' +
                ", handlerKey='" + handlerKey + '\'' +
                ", success=" + success +
                ", duration=" + duration +
                '}';
    }
}
