package com.atlassian.jira.event.user.anonymize;

import com.atlassian.analytics.api.annotations.EventName;

import java.util.Objects;

/**
 * This event is sent when a user has been anonymized.
 *
 * @since v8.4
 */
@EventName("jira.user.anonymized")
public class UserAnonymizationFinishedEvent {
    private final String userKey;
    private final String userName;
    private final String oldUserKey;
    private final String oldUserName;
    private final boolean reRun;
    private final String reRunUserKey;
    private final String reRunUserName;

    public UserAnonymizationFinishedEvent(String userKey, String userName, String oldUserKey, String oldUserName, boolean reRun, String reRunUserKey, String reRunUserName) {
        this.userKey = userKey;
        this.userName = userName;
        this.oldUserKey = oldUserKey;
        this.oldUserName = oldUserName;
        this.reRun = reRun;
        this.reRunUserKey = reRunUserKey;
        this.reRunUserName = reRunUserName;
    }

    /**
     * User key after anonymization process (current value).
     *
     * @return User key after anonymization process.
     */
    public String getUserKey() {
        return userKey;
    }

    /**
     * Username after anonymization process (current value).
     *
     * @return Username after anonymization process.
     */
    public String getUserName() {
        return userName;
    }

    /**
     * User key before anonymization process.
     *
     * @return User key before anonymization process.
     */
    public String getOldUserKey() {
        return oldUserKey;
    }

    /**
     * Username before anonymization process.
     *
     * @return Username before anonymization process.
     */
    public String getOldUserName() {
        return oldUserName;
    }

    /**
     * Set to true when re-run was requested.
     *
     * @return true if re-run was requested. false otherwise.
     */
    public boolean isReRun() {
        return reRun;
    }

    /**
     * Old user key provided for re-run.
     *
     * @return Old user key provided for rerun
     */
    public String getReRunUserKey() {
        return reRunUserKey;
    }

    /**
     * Old username provided for re-run.
     *
     * @return Old user key provided for rerun.
     */
    public String getReRunUserName() {
        return reRunUserName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserAnonymizationFinishedEvent that = (UserAnonymizationFinishedEvent) o;
        return reRun == that.reRun &&
                Objects.equals(userKey, that.userKey) &&
                Objects.equals(userName, that.userName) &&
                Objects.equals(oldUserKey, that.oldUserKey) &&
                Objects.equals(oldUserName, that.oldUserName) &&
                Objects.equals(reRunUserKey, that.reRunUserKey) &&
                Objects.equals(reRunUserName, that.reRunUserName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userKey, userName, oldUserKey, oldUserName, reRun, reRunUserKey, reRunUserName);
    }

    @Override
    public String toString() {
        return "UserAnonymizationFinishedEvent{" +
                "userKey='" + userKey + '\'' +
                ", userName='" + userName + '\'' +
                ", oldUserKey='" + oldUserKey + '\'' +
                ", oldUserName='" + oldUserName + '\'' +
                ", reRun=" + reRun +
                ", reRunUserKey='" + reRunUserKey + '\'' +
                ", reRunUserName='" + reRunUserName + '\'' +
                '}';
    }
}
