package com.atlassian.jira.event.user;

import com.atlassian.annotations.ExperimentalApi;
import com.google.common.base.MoreObjects;

import java.util.Objects;
import java.util.Set;

import static java.util.Collections.singleton;
import static java.util.Collections.unmodifiableSet;

/**
 * Event that indicates a batch of new Users have been added to JIRA.
 * <p>
 * Unlike {@link UserEvent} this will only and always be fired, whenever atleast 1 new User is added to the database.
 *
 * @since v7.4.0
 */
@ExperimentalApi
public final class UsersAddedEvent {

    private final Set<AddedUser> addedUsers;

    public UsersAddedEvent(AddedUser addedUser) {
        this(singleton(addedUser));
    }

    public UsersAddedEvent(Set<AddedUser> addedUsers) {
        this.addedUsers = unmodifiableSet(addedUsers);
    }

    public Set<AddedUser> getAddedUsers() {
        return addedUsers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UsersAddedEvent that = (UsersAddedEvent) o;
        return Objects.equals(addedUsers, that.addedUsers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(addedUsers);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("addedUsers", addedUsers)
                .toString();
    }

    @ExperimentalApi
    public static final class AddedUser {
        private final String userKey;
        private final String emailAddress;

        public AddedUser(String userKey, String emailAddress) {
            this.userKey = userKey;
            this.emailAddress = emailAddress;
        }

        public String getUserKey() {
            return userKey;
        }

        public String getEmailAddress() {
            return emailAddress;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            AddedUser addedUser = (AddedUser) o;
            return Objects.equals(userKey, addedUser.userKey) &&
                    Objects.equals(emailAddress, addedUser.emailAddress);
        }

        @Override
        public int hashCode() {
            return Objects.hash(userKey, emailAddress);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("userKey", userKey)
                    .add("emailAddress", emailAddress)
                    .toString();
        }
    }
}
