package com.atlassian.jira.event.operation;

import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.Nonnull;
import java.util.concurrent.Callable;
import java.util.function.Supplier;

/**
 * Allows to execute some arbitrary code as a part of larger "spanning" operation.
 *
 * @since 7.13
 */
@ExperimentalApi
public interface SpanningOperationRunner {
    /**
     * Allows marking events generated by passed {@code callable} with passed {@code surroundingOperation}.
     * Current implementation allows setting spanning operation for current thread only.
     * If some other thread is spawned in the passed {@code callable} then spanning operation has to be set up for the newly started thread.
     *
     * Currently only events that implement {@link SpanningOperationEvent} are affected by this method.
     *
     * @param spanningOperation Operation that spans over passed {@code supplier}
     * @param supplier          Set of instructions executed in the context of spanning operation
     * @param <T>               Type of returned object
     * @return value returned by passed {@code supplier}
     */
    <T> T execute(@Nonnull SpanningOperation spanningOperation, @Nonnull Supplier<T> supplier);

    /**
     * Allows marking events generated by passed {@code runnable} with passed {@code surroundingOperation}
     * Current implementation allows setting spanning operation for current thread only.
     * If some other thread is spawned in the passed {@code runnable} then spanning operation has to be set up for the newly started thread.
     *
     * Currently only events that implement {@link SpanningOperationEvent} are affected by this method.
     *
     * @param spanningOperation Operation that spans over passed {@code runnable}
     * @param runnable
     */
    void execute(@Nonnull SpanningOperation spanningOperation, @Nonnull Runnable runnable);

    /**
     * Allows marking events generated by passed {@code callable} with passed {@code surroundingOperation}.
     * Current implementation allows setting spanning operation for current thread only.
     * If some other thread is spawned in the passed {@code callable} then spanning operation has to be set up for the newly started thread.
     *
     * Currently only events that implement {@link SpanningOperationEvent} are affected by this method.
     *
     * Exceptions thrown by {@code callable} are converted to {@link RuntimeException} and re-thrown.
     *
     * @param spanningOperation Operation that spans over passed {@code callable}
     * @param callable          Set of instructions executed in the context of spanning operation
     * @param <T>               Type of returned object
     * @return value returned by passed {@code callable}
     */
    <T> T executeCallable(@Nonnull SpanningOperation spanningOperation, @Nonnull Callable<T> callable);

}
