package com.atlassian.jira.event.operation;

import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Objects;
import java.util.UUID;

/**
 * Holds information about event being part of one logical operation eg. event can be part of bulk move, bulk edit, bulk issue create.
 * Some spanning operation can be part of larger operation - in this case {@code parent} will be set.
 *
 * @since 7.13
 */
@ExperimentalApi
public class SpanningOperation implements Serializable {
    private final SpanningOperation parent;
    private final String type;
    private final String id;

    /**
     * @param parent Parent operation
     * @param type   Type of operation eg. bulk move, bulk edit.
     * @param id     Id of operation, should be used to discriminate distinct operations from each other.
     */
    private SpanningOperation(@Nullable SpanningOperation parent, @Nonnull String type, @Nonnull String id) {
        this.parent = parent;
        this.type = Objects.requireNonNull(type);
        this.id = Objects.requireNonNull(id);
    }

    private SpanningOperation(Builder builder) {
        this(builder.parent, builder.type, builder.id);
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * @return Parent surrounding operation
     */
    public SpanningOperation getParent() {
        return parent;
    }

    /**
     * Type of surrounding operation eg. bulk move, bulk edit etc.
     *
     * @return type of operation
     */
    @Nonnull
    public String getType() {
        return type;
    }

    /**
     * Id of surrounding operation. Should be used to distinguish one spanning operation from another.
     *
     * @return id of operation
     */
    @Nonnull
    public String getId() {
        return id;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SpanningOperation that = (SpanningOperation) o;
        return Objects.equals(parent, that.parent) &&
                Objects.equals(type, that.type) &&
                Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(parent, type, id);
    }

    @Override
    public String toString() {
        return "SpanningOperation{" +
                "parent=" + parent +
                ", type='" + type + '\'' +
                ", id='" + id + '\'' +
                '}';
    }


    public static final class Builder {
        private SpanningOperation parent;
        private String type;
        private String id;

        private Builder() {
        }

        public Builder parent(SpanningOperation parent) {
            this.parent = parent;
            return this;
        }

        public Builder type(String type) {
            this.type = type;
            return this;
        }

        public Builder type(SpanningOperationType type) {
            this.type = type.toString();
            return this;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder generatedId() {
            this.id = UUID.randomUUID().toString();
            return this;
        }

        public SpanningOperation build() {
            return new SpanningOperation(this);
        }
    }
}
