package com.atlassian.jira.event.issue;

import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.notification.NotificationRecipient;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;

import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;

/**
 * When a user mentions another user on an issue via the @username or [~username] syntax this event will be fired. It
 * contains the user that added the comment or description, all users mentioned, the issue as well as the text in which
 * the mentions occurred. Finally it also contains the fieldId of the field in which users were mentioned.
 *
 * @since v5.0
 */
@PublicApi
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public class MentionIssueEvent {
    private final Issue issue;
    private final ApplicationUser fromUser;
    private final Set<ApplicationUser> toUsers;
    private final Set<NotificationRecipient> recipients;
    private final String mentionText;
    private final String fieldId;
    private final Set<NotificationRecipient> currentRecipients;

    @Deprecated
    public MentionIssueEvent(Issue issue,
                             ApplicationUser fromUser,
                             Set<ApplicationUser> toUsers,
                             String mentionText,
                             String fieldId,
                             @Nullable Set<NotificationRecipient> currentRecipients) {
        this(issue, fromUser, toUsers, emptySet(), mentionText, fieldId, currentRecipients);
    }

    public MentionIssueEvent(Issue issue,
                             ApplicationUser fromUser,
                             Set<ApplicationUser> toUsers,
                             Set<NotificationRecipient> recipients,
                             String mentionText,
                             String fieldId,
                             @Nullable Set<NotificationRecipient> currentRecipients) {
        this.issue = requireNonNull(issue);
        this.fromUser = requireNonNull(fromUser);
        this.toUsers = requireNonNull(toUsers);
        this.recipients = requireNonNull(recipients);
        this.mentionText = requireNonNull(mentionText);
        this.fieldId = requireNonNull(fieldId);
        this.currentRecipients = currentRecipients;
    }

    public Issue getIssue() {
        return issue;
    }

    public ApplicationUser getFromUser() {
        return fromUser;
    }

    /**
     * Returns the set of mentioned users, after permission checks.
     */
    public Set<ApplicationUser> getToUsers() {
        return Collections.unmodifiableSet(toUsers);
    }

    /**
     * Returns the recipients that should receive an email for this mention, after de-duplicating users that already
     * received an email for an action in this bundle.
     *
     * @since 8.9
     */
    public Set<NotificationRecipient> getRecipients() {
        return recipients;
    }

    public String getMentionText() {
        return mentionText;
    }

    public String getFieldId() {
        return fieldId;
    }

    /**
     * Returns the recipients that are already receiving an email for the event that triggered this mention (such as issue commented event, etc.).
     */
    @Deprecated
    @Nullable
    public Set<NotificationRecipient> getCurrentRecipients() {
        return currentRecipients;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        MentionIssueEvent that = (MentionIssueEvent) o;

        return Objects.equals(issue, that.issue) &&
                Objects.equals(fromUser, that.fromUser) &&
                Objects.equals(toUsers, that.toUsers) &&
                Objects.equals(recipients, that.recipients) &&
                Objects.equals(mentionText, that.mentionText) &&
                Objects.equals(fieldId, that.fieldId) &&
                Objects.equals(currentRecipients, that.currentRecipients);
    }

    @Override
    public int hashCode() {
        return Objects.hash(issue, fromUser, toUsers, recipients, mentionText, fieldId, currentRecipients);
    }

    @Override
    public String toString() {
        return "MentionIssueEvent{" +
                "issue=" + issue +
                ", fromUser=" + fromUser +
                ", toUsers=" + toUsers +
                ", recipients=" + recipients +
                ", mentionText='" + mentionText + '\'' +
                ", fieldId='" + fieldId + '\'' +
                ", currentRecipients=" + currentRecipients +
                '}';
    }
}
