package com.atlassian.jira.event.issue;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.event.AbstractEvent;
import com.atlassian.jira.event.operation.SpanningOperation;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.user.ApplicationUser;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import javax.annotation.Nullable;
import java.util.Optional;

/**
 * IssuePreDeleteEvent is triggered before an issue is deleted.
 *
 * @since 5.2.6
 */
@PublicApi
public final class IssuePreDeleteEvent extends AbstractEvent implements IssueRelatedEvent {
    private final Issue issue;
    private final ApplicationUser user;
    private final SpanningOperation spanningOperation;

    public IssuePreDeleteEvent(final Issue issue, final ApplicationUser user) {
        this(issue, user, null);
    }

    /**
     * @param issue             issue
     * @param user              user
     * @param spanningOperation Spanning operation - operation this event is part of.
     * @since 7.13
     */
    public IssuePreDeleteEvent(final Issue issue, final ApplicationUser user, final @Nullable SpanningOperation spanningOperation) {
        this.issue = issue;
        this.user = user;
        this.spanningOperation = spanningOperation;
    }

    @Override
    public Issue getIssue() {
        return issue;
    }

    public ApplicationUser getUser() {
        return user;
    }

    @Override
    public Optional<SpanningOperation> getSpanningOperation() {
        return Optional.ofNullable(spanningOperation);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof IssuePreDeleteEvent) {
            final IssuePreDeleteEvent that = (IssuePreDeleteEvent) o;
            return super.equals(that)
                    && Objects.equal(this.user, that.user)
                    && Objects.equal(this.issue, that.issue)
                    && Objects.equal(this.spanningOperation, that.spanningOperation);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(super.hashCode(), issue, user, spanningOperation);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("time", time)
                .add("params", params)
                .add("issue", issue)
                .add("user", user)
                .add("spanningOperation", spanningOperation)
                .toString();
    }
}
