package com.atlassian.jira.event.commit;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.event.issue.IssueEventManager;
import com.atlassian.jira.event.commit.OnCommitEvent.OnCommitEventDataReloadStatus;
import com.atlassian.jira.event.issue.commit.OnCommitIssueEventBundle;
import com.atlassian.jira.util.InjectableComponent;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.function.Supplier;

/**
 * Allow to publish {@link OnCommitEvent} and {@link OnCommitIssueEventBundle} events, that if inside a transaction will
 * not actually be published until the transaction has finished. If not inside a transaction, the events will be
 * published immediately.
 * <p>
 * NB: If you would like to fire an issue related event, it is recommended to use
 * {@link IssueEventManager#dispatchIssueEventOnCommit(Supplier)} or
 * {@link IssueEventManager#dispatchIssueEventBundleOnCommit(Supplier)} as these will provide better semantics for
 * the wrapped issue specific events.
 *
 * @see OnCommitEvent
 * @see IssueEventManager
 * @since v7.4.0
 */
@ExperimentalApi
@InjectableComponent
@ParametersAreNonnullByDefault
public interface OnCommitEventManager {

    /**
     * Will wrap the provided event into a default {@link OnCommitEvent} object, for publishing.
     * <p>
     * The {@link Supplier} will be invoked when the event is about to be published.
     * <p>
     * NB: This is intended for non issue related events that would like to ensure they are fired after any transaction
     * has completed. For issue related {@link OnCommitEvent}, use {@link IssueEventManager}
     *
     * @param eventSupplier The supplier of the event object to wrap into an {@link OnCommitEvent} event
     */
    void dispatchAsOnCommitEvent(Supplier<Object> eventSupplier);

    /**
     * Will publish the event immediately if not inside a transaction, otherwise will be published once the
     * current transactions has finished.
     * <p>
     * The {@link Supplier} will be invoked when the event is about to be published, so any data that should be reloaded
     * can be done inside it. Potentially the {@link OnCommitEvent#getOnCommitEventDataReloadStatus()} can be set to
     * {@link OnCommitEventDataReloadStatus#RELOADED_ON_COMMIT} for the event.
     * <p>
     * NB: This is intended for non issue related events that would like to ensure they are fired after any transaction
     * has completed. For issue related {@link OnCommitEvent}, use {@link IssueEventManager}
     *
     * @param onCommitEventSupplier The supplier of the {@link OnCommitEvent} event to send
     * @param <T>                   The wrapped type of the event
     */
    <T> void dispatchOnCommitEvent(Supplier<OnCommitEvent<T>> onCommitEventSupplier);

}
