package com.atlassian.jira.event;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.permission.GlobalPermissionKey;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.security.GlobalPermissionManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.workflow.JiraWorkflow;
import com.atlassian.jira.workflow.ProjectWorkflowSchemeHelper;
import com.google.common.base.Preconditions;

import javax.annotation.Nonnull;
import java.util.Collection;

/**
 * Author: jdoklovic
 */
public abstract class AbstractWorkflowEvent extends AbstractEvent {

    private final JiraWorkflow workflow;
    private final boolean isAdmin;
    private final boolean isSysAdmin;
    private final boolean isolated;
    private final boolean hasExtPermission;
    private final Long projectId;

    public AbstractWorkflowEvent(@Nonnull JiraWorkflow workflow) {
        super();
        Preconditions.checkNotNull(workflow, "workflow must not be null");
        this.workflow = workflow;

        final ApplicationUser user = ComponentAccessor.getJiraAuthenticationContext().getLoggedInUser();
        final GlobalPermissionManager globalPermissionManager = ComponentAccessor.getGlobalPermissionManager();

        this.isAdmin = globalPermissionManager.hasPermission(GlobalPermissionKey.ADMINISTER, user);
        this.isSysAdmin = globalPermissionManager.hasPermission(GlobalPermissionKey.SYSTEM_ADMIN, user);
        final ProjectWorkflowSchemeHelper projectWorkflowSchemeHelper = ComponentAccessor.getComponent(ProjectWorkflowSchemeHelper.class);
        this.isolated = projectWorkflowSchemeHelper.isWorkflowIsolated(workflow.getName());
        if (this.isolated) {
            final Collection<Project> workflowUsedInProjects = projectWorkflowSchemeHelper.getAllProjectsForWorkflow(workflow.getName());
            if (workflowUsedInProjects.size() == 1) {
                final Project workflowProject = workflowUsedInProjects.iterator().next();
                this.projectId = workflowProject.getId();
                this.hasExtPermission = ComponentAccessor.getPermissionSchemeManager().hasExtendedProjectAdministration(workflowProject);
            } else {
                this.projectId = null;
                this.hasExtPermission = false;
            }
        } else {
            this.projectId = null;
            this.hasExtPermission = false;
        }
    }

    public JiraWorkflow getWorkflow() {
        return workflow;
    }

    public int getWorkflowHash() {
        return workflow.getName().hashCode();
    }

    public boolean getIsAdmin() {
        return isAdmin;
    }

    public boolean getIsSysAdmin() {
        return isSysAdmin;
    }

    public boolean getIsolated() {
        return isolated;
    }

    public boolean isHasExtPermission() {
        return hasExtPermission;
    }

    public Long getProjectId() {
        return projectId;
    }
}
