package com.atlassian.jira.bc.project;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.permission.PermissionSchemeManager;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.security.Permissions;
import com.atlassian.jira.security.plugin.ProjectPermissionKey;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import java.util.function.Predicate;

import static com.atlassian.jira.permission.ProjectPermissions.ADMINISTER_PROJECTS;
import static com.atlassian.jira.security.Permissions.ADMINISTER;

/**
 * Represents the different actions a users wants to perform on a project.
 */
public enum ProjectAction {
    /**
     * The user is able to see the project. This does not mean the user can edit the project or even view its issues.
     */
    VIEW_PROJECT(new int[]{ADMINISTER, Permissions.BROWSE, Permissions.PROJECT_ADMIN}, true,
            "admin.errors.project.no.view.permission"),

    /**
     * The user is able to see the archived project. This does not mean the user can edit the project or even view its issues.
     */
    VIEW_ARCHIVED_PROJECT(new int[]{ADMINISTER, Permissions.BROWSE, Permissions.PROJECT_ADMIN}, false,
            "admin.errors.project.no.view.permission"),

    /**
     * Able to view the issues for the passed project.
     */
    VIEW_ISSUES(new int[]{Permissions.BROWSE}, false,
            "admin.errors.project.no.browse.permission"),

    /**
     * Able to configure the project specific configuration.
     */
    EDIT_PROJECT_CONFIG(new int[]{ADMINISTER, Permissions.PROJECT_ADMIN}, true,
            "admin.errors.project.no.config.permission"),

    /**
     * Able to see the project specific configuration.
     */
    VIEW_PROJECT_CONFIG(new int[]{ADMINISTER, Permissions.PROJECT_ADMIN}, false,
            "admin.errors.project.no.config.permission"),

    /**
     * Able to archive the project.
     */
    ARCHIVE_PROJECT(new int[]{ADMINISTER}, false,
            "admin.errors.project.no.archive.permission"),

    /**
     * Able to delete the project.
     */
    DELETE_PROJECT(new int[]{ADMINISTER, Permissions.PROJECT_ADMIN}, false,
            "admin.errors.project.no.delete.permission"),

    /**
     * Able to configure the project specific configuration with the project key.
     */
    EDIT_PROJECT_KEY(new int[]{ADMINISTER}, true,
            "admin.projects.service.error.no.admin.permission.key"),

    /**
     * Able to configure the project specific configuration which requires "extended project admin" permission.
     */
    EDIT_PROJECT_CONFIG_EXTENDED(ProjectAction::isExtendedProjectAdmin, true,
            "admin.errors.project.no.extended.config.permission");

    private final int[] permissions;
    private final String errorKey;
    private final Predicate<PredicateContext> permissionPredicate;
    private final boolean openProjectRequired;

    ProjectAction(int[] permissions, boolean openProjectRequired, String errorKey) {
        this.permissions = permissions;
        this.errorKey = errorKey;
        this.permissionPredicate = null;
        this.openProjectRequired = openProjectRequired;
    }

    ProjectAction(Predicate<PredicateContext> permissionPredicate, boolean openProjectRequired, String errorKey) {
        this.permissions = null;
        this.errorKey = errorKey;
        this.permissionPredicate = permissionPredicate;
        this.openProjectRequired = openProjectRequired;
    }

    int[] getPermissions() {
        return permissions;
    }

    public String getErrorKey() {
        return errorKey;
    }

    public boolean hasPermissionPredicate() {
        return permissionPredicate != null;
    }

    public boolean hasPermission(PermissionManager permissionManager, ApplicationUser user, Project project) {
        final PredicateContext predicateContext = new PredicateContext(
                permissionManager,
                user,
                project,
                ComponentAccessor.getPermissionSchemeManager());

        if (openProjectRequired && project.isArchived()) {
            return false;
        } else if (hasPermissionPredicate()) {
            return permissionPredicate.test(predicateContext);
        } else {
            return checkPermissions(predicateContext);
        }
    }

    private boolean checkPermissions(PredicateContext predicateContext) {
        for (int permission : permissions) {
            if (Permissions.isGlobalPermission(permission)) {
                if (predicateContext.getPermissionManager().hasPermission(permission, predicateContext.getUser())) {
                    return true;
                }
            } else if (predicateContext.getPermissionManager().hasPermission(new ProjectPermissionKey(permission),
                    predicateContext.getProject(), predicateContext.getUser())) {
                return true;
            }
        }
        return false;
    }

    static Boolean isExtendedProjectAdmin(@Nonnull PredicateContext context) {
        if (isGlobalAdmin(context)) {
            return true;
        }

        return isProjectAdmin(context) && context.getPermissionSchemeManager().hasExtendedProjectAdministration(context.getProject());
    }

    private static Boolean isGlobalAdmin(@Nonnull PredicateContext context) {
        return context.getPermissionManager().hasPermission(ADMINISTER, context.getUser());
    }

    private static Boolean isProjectAdmin(@Nonnull PredicateContext context) {
        return context.getPermissionManager().hasPermission(ADMINISTER_PROJECTS, context.getProject(), context.getUser());
    }


    static class PredicateContext {
        private final PermissionManager permissionManager;
        private final ApplicationUser user;
        private final Project project;
        private final PermissionSchemeManager permissionSchemeManager;

        PredicateContext(PermissionManager manager, ApplicationUser user, Project project,
                         PermissionSchemeManager permissionSchemeManager) {
            this.permissionManager = manager;
            this.user = user;
            this.project = project;
            this.permissionSchemeManager = permissionSchemeManager;
        }

        public PermissionManager getPermissionManager() {
            return permissionManager;
        }

        public ApplicationUser getUser() {
            return user;
        }

        public Project getProject() {
            return project;
        }

        public PermissionSchemeManager getPermissionSchemeManager() {
            return permissionSchemeManager;
        }
    }
}
