package com.atlassian.jira.bc.issue.fields.screen;

import com.atlassian.annotations.ExperimentalApi;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

import java.util.Optional;

/**
 * Represents intention to move a field along the screen
 *
 * @since v7.4
 */
@ExperimentalApi
@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
public class MoveFieldInfo {

    public enum MoveFieldPosition {
        First,
        Last
    }

    private final Optional<String> afterField;
    private final Optional<MoveFieldPosition> desiredPosition;

    /**
     * Creates new instance of move intent. It should contain one and only one actual intent.
     *
     * @param afterField      expresses an intent of move given field after already existing field
     * @param desiredPosition expresses intent of move given field disregarding other fields
     */
    public MoveFieldInfo(Optional<String> afterField, Optional<MoveFieldPosition> desiredPosition) {
        this.afterField = afterField;
        this.desiredPosition = desiredPosition;
    }

    public Optional<String> getAfterField() {
        return afterField;
    }

    public Optional<MoveFieldPosition> getDesiredPosition() {
        return desiredPosition;
    }

    /**
     * Creates a new instance with intent of moving after some other field on given tab.
     *
     * @param field a field's ID
     * @return a new move intent
     */
    public static MoveFieldInfo after(String field) {
        Preconditions.checkArgument(StringUtils.isNotBlank(field), "Field ID should not be blank");
        return new MoveFieldInfo(Optional.of(field), Optional.empty());
    }

    /**
     * Creates a new instance with intent of moving given field absolutely
     *
     * @param position desired position
     * @return a new move intent
     */
    public static MoveFieldInfo ofPosition(MoveFieldPosition position) {
        return new MoveFieldInfo(Optional.empty(), Optional.ofNullable(position));
    }
}
