package com.atlassian.jira.bc.issue.fields.screen;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Optional;

/**
 * A representation of a context in which current screen operation is being executed. It always consists of user and optionally may also involve project.
 *
 * @since v7.4
 */
@ExperimentalApi
@ParametersAreNonnullByDefault
@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
public class FieldScreenOperationContext {


    private final ApplicationUser user;
    private final Optional<Project> project;

    private FieldScreenOperationContext(final ApplicationUser user, final Optional<Project> project) {
        this.user = Preconditions.checkNotNull(user);
        this.project = Preconditions.checkNotNull(project);
    }

    /**
     * @return a user performing operation
     */
    public ApplicationUser getUser() {
        return user;
    }


    /**
     * @return a project in which context operation is being performed. may be empty
     */
    public Optional<Project> getProject() {
        return project;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        FieldScreenOperationContext that = (FieldScreenOperationContext) o;

        if (!user.equals(that.user)) return false;
        return project.equals(that.project);
    }

    @Override
    public int hashCode() {
        int result = user.hashCode();
        result = 31 * result + project.hashCode();
        return result;
    }

    /**
     * Creates user and project context.
     *
     * @param user    a user performing operation
     * @param project a project in which context operation is being performed
     * @return
     */
    public static FieldScreenOperationContext ofUserAndProject(final ApplicationUser user, @Nullable final Project project) {
        return new FieldScreenOperationContext(user, Optional.ofNullable(project));
    }

    /**
     * Creates new user-only context. Project is not present
     *
     * @param user user performing operation
     * @return a new context
     */
    public static FieldScreenOperationContext ofUser(final ApplicationUser user) {
        return new FieldScreenOperationContext(user, Optional.empty());
    }
}
