package com.atlassian.jira.bc.issue.fields.screen;

import com.atlassian.annotations.ExperimentalApi;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * An object expressing identifier of {@link com.atlassian.jira.issue.fields.screen.FieldScreenLayoutItem}.
 * In essence, identifier is a tuple of screen tab's identifier and field's system identifier
 *
 * @see FieldScreenTabId
 * @see FieldScreenId
 * @since v7.4
 */
@ExperimentalApi
@ParametersAreNonnullByDefault
public class FieldScreenLayoutItemId {
    private final FieldScreenTabId tabId;
    private final String fieldId;


    private FieldScreenLayoutItemId(final FieldScreenTabId tabId, final String fieldId) {
        this.tabId = Preconditions.checkNotNull(tabId);
        this.fieldId = Preconditions.checkNotNull(fieldId);
    }

    /**
     * Creates identifier based on tab's identifier and field's identifier
     *
     * @param tabId   identifier of given {@link com.atlassian.jira.issue.fields.screen.FieldScreenTab}
     * @param fieldId ID of field
     * @return identifier object
     */
    public static FieldScreenLayoutItemId byIds(final FieldScreenTabId tabId, final String fieldId) {
        return new FieldScreenLayoutItemId(tabId, fieldId);
    }

    public FieldScreenTabId getTabId() {
        return tabId;
    }

    public String getFieldId() {
        return fieldId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        FieldScreenLayoutItemId that = (FieldScreenLayoutItemId) o;

        if (!tabId.equals(that.tabId)) return false;
        return fieldId.equals(that.fieldId);
    }

    @Override
    public int hashCode() {
        int result = tabId.hashCode();
        result = 31 * result + fieldId.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("tabId", tabId)
                .append("fieldId", fieldId)
                .toString();
    }
}
