package com.atlassian.jira.bc.filter;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.bc.JiraServiceContext;
import com.atlassian.jira.issue.search.SearchRequest;
import com.atlassian.jira.issue.subscription.FilterSubscription;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Date;

/**
 * Provides high level access to CRUD and query FilterSubscriptions.
 */
@PublicApi
public interface FilterSubscriptionService {
    /**
     * Validates a given cron expression
     * <p>
     * Errors are passed back in the {@link com.atlassian.jira.util.ErrorCollection}
     * of the {@link com.atlassian.jira.bc.JiraServiceContext}
     *
     * @param context Jira service context
     * @param expr    expression to evaluate
     */
    void validateCronExpression(JiraServiceContext context, String expr);

    /**
     * Create and store a subscription that uses the given cron expression
     *
     * @param context      Jira service context
     * @param filterId     id of the filter subscribing to
     * @param groupName    sent to group (may be null if sent to self)
     * @param expr         cron expression to store
     * @param emailOnEmpty send email if filter returns no results
     */
    void storeSubscription(JiraServiceContext context, Long filterId, String groupName, String expr, boolean emailOnEmpty);

    /**
     * Updates the subscription to the new given values and persists.
     * <p>
     * <b>Note: This method will always change the owner of the subscription to the logged in user.</b> If you don't want
     * this behavior, use {@link #updateSubscription(JiraServiceContext, ApplicationUser, Long, String, String, boolean)}.
     *
     * @param context        Jira service context
     * @param subscriptionId id of the subscription being updated
     * @param groupName      sent to group (may be null if sent to self)
     * @param expr           cron expression to store
     * @param emailOnEmpty   send email if filter returns no results
     */
    void updateSubscription(JiraServiceContext context, Long subscriptionId, String groupName, String expr, boolean emailOnEmpty);

    /**
     * Updates the subscription to the new given values and persists.
     *
     * @param context        Jira service context
     * @param owner          new owner for the subscription
     * @param subscriptionId id of the subscription being updated
     * @param groupName      sent to group (may be null if sent to self)
     * @param expr           cron expression to store
     * @param emailOnEmpty   send email if filter returns no results
     * @since v8.11
     */
    void updateSubscription(JiraServiceContext context, ApplicationUser owner, Long subscriptionId, String groupName, String expr, boolean emailOnEmpty);

    /**
     * Renders a human readable description of the given cron expression.
     * <p>
     * Returns the cron expression if it can't be parsed by the {@link com.atlassian.core.cron.parser.CronExpressionParser}.
     *
     * @param context        Jira service context
     * @param cronExpression a cron expression.
     * @return a locale-specific sentence describing the cron string (or on failure, the cron string).
     */
    String getPrettySchedule(JiraServiceContext context, String cronExpression);

    /**
     * Retrieves subscriptions that a user can see for a filter.
     * <p>
     * An admin can see all subscriptions, the filter owner can see all subscriptions for their filter, otherwise
     * a user can only see their own subscriptions.
     *
     * @param user   the user that can see the subscriptions
     * @param filter the filter with the associated subscriptions
     * @return a collection of subscriptions
     */
    Collection<FilterSubscription> getVisibleFilterSubscriptions(ApplicationUser user, SearchRequest filter);

    /**
     * Retrieve the cron expression associated with this subscription
     *
     * @param subscription
     * @return the cron expression associated with this subscription
     */
    String getCronExpression(JiraServiceContext context, FilterSubscription subscription);

    /**
     * Returns the next send time for this subscription.
     * <p>
     * This may return null if the scheduler does not support the reporting of next send times.
     *
     * @param subscription the subscription
     * @return next send time
     */
    @Nullable
    Date getNextSendTime(@Nonnull FilterSubscription subscription);
}
