package com.atlassian.jconnect.jira.customfields;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.converters.DoubleConverter;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.indexers.impl.AbstractCustomFieldIndexer;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * Indexer for location custom field.
 *
 * @see LocationCFType
 */
public class LocationIndexer extends AbstractCustomFieldIndexer {

    public static String latFieldId(String locationFieldId) {
        return locationFieldId + "-lat";
    }

    public static String lngFieldId(String locationFieldId) {
        return locationFieldId + "-lng";
    }

    private final CustomField field;
    private final DoubleConverter doubleConverter;

    public LocationIndexer(FieldVisibilityManager fieldVisibilityManager, CustomField customField,
                           DoubleConverter doubleConverter) {
        super(fieldVisibilityManager, notNull("field", customField));
        this.field = customField;
        this.doubleConverter = doubleConverter;
    }


    @Override
    public void addDocumentFieldsSearchable(Document doc, Issue issue) {
        addDocumentFields(doc, issue, Field.Index.NOT_ANALYZED);
    }

    @Override
    public void addDocumentFieldsNotSearchable(Document doc, Issue issue) {
        addDocumentFields(doc, issue, Field.Index.NO);
    }

    // TODO this sucks, we should be using NumericField (as of JIRA 4.4)

    private void addDocumentFields(Document doc, Issue issue, Field.Index index) {
        Object value = field.getValue(issue);
        if (value != null) {
            // TODO we need to normalize cause those guys do not support negative numbers. hopefully NumericField does!
            final Location location = ((Location) value).normalize();
            doc.add(new Field(latFieldId(getDocumentFieldId()), doubleConverter.getStringForLucene(location.lat), Field.Store.YES, index));
            doc.add(new Field(lngFieldId(getDocumentFieldId()), doubleConverter.getStringForLucene(location.lng), Field.Store.YES, index));
        }
    }

}
