package com.atlassian.integrationtesting.ui;

import java.io.File;
import java.io.IOException;

import com.atlassian.integrationtesting.Functions;
import com.atlassian.integrationtesting.ui.CompositeUiTester.Backup;
import com.atlassian.integrationtesting.ui.CompositeUiTester.Login;
import com.atlassian.integrationtesting.ui.CompositeUiTester.WebSudoLogin;
import com.atlassian.sal.api.ApplicationProperties;

import com.gargoylesoftware.htmlunit.html.HtmlPage;
import com.google.common.base.Function;

import be.roam.hue.doj.Doj;

import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * Provides {@code UiTester} implementations for a selection of known application versions.
 */
public class UiTesters
{
    public static UiTester newUiTester(ApplicationProperties applicationProperties, UiTesterFunctionProvider functions)
    {
        return new CompositeUiTester(applicationProperties,
                functions.logIn(), 
                functions.webSudoLogIn(), 
                functions.logout(), 
                functions.getLoggedInUser(), 
                functions.isOnLogInPage(),
                functions.restore());
    }
    
    /**
     * Returns a function that expects the ait-plugin to be installed and the rest/ait/1.0/user resource available.
     * @return a funtion that expects the ait-plugin to be installed and the rest/ait/1.0/user resource available
     */
    public static Function<UiTester, String> getLoggedInUser()
    {
        return GetLoggedInUser.INSTANCE;
    }
    
    private enum GetLoggedInUser implements Function<UiTester, String>
    {
        INSTANCE;

        public String apply(UiTester uiTester)
        {
            return Doj.on(uiTester.gotoPage("rest/ait/1.0/user")).getById("user").text();
        }
    }
    
    /**
     * Returns a function that simply returns the current page.
     * 
     * @return a function that simply returns the current page
     */
    public static Function<WebSudoLogin, HtmlPage> doNothingWebSudoLogin()
    {
        return DoNothingWebSudoLogin.INSTANCE;
    }
    
    private enum DoNothingWebSudoLogin implements Function<WebSudoLogin, HtmlPage>
    {
        INSTANCE;

        public HtmlPage apply(WebSudoLogin login)
        {
            return (HtmlPage) login.client.currentPage().firstElement().getPage();
        }
    }
    
    /**
     * Returns a function that logs a user out by going a GET to the specified {@code page}.
     * 
     * @param page Path of the logout page relative to the base url
     * @return a function that logs a user out by going a GET to the specified {@code page}
     */
    public static Function<UiTester, Void> goToPageToLogout(String page)
    {
        return new GoToPageToLogout(page);
    }
    
    private static final class GoToPageToLogout implements Function<UiTester, Void>
    {
        private final String page;
        
        GoToPageToLogout(String page)
        {
            this.page = page;
        }
        
        public Void apply(UiTester uiTester)
        {
            uiTester.gotoPage(page);
            return null;
        }
    }

    /**
     * Returns a builder that allows you to create a function goes to {@code page} and manipulates the log in form.
     * 
     * @param page Path of the log in page relative to the base url
     * @return a builder that allows you to create a function goes to {@code page} and manipulates the log in form
     */
    public static LogInFunctionBuilder logInByGoingTo(String page)
    {
        return new LogInFunctionBuilder(page);
    }
    
    /**
     * A builder that allows you to create a function will go to the log in page, fill out the log in details, and log
     * a user in.  Instances of this class are immutable and may be used across threads and reused to build other 
     * functions. 
     */
    public static final class LogInFunctionBuilder
    {
        private final String page;
        private final String formName;
        private final String formId;
        private final String userInputName;
        private final String passwordInputName;
        private final String submitButtonId;
        private final String submitButtonName;
        
        private LogInFunctionBuilder(String page)
        {
            this(page, null, null, "os_username", "os_password", null, null);
        }

        private LogInFunctionBuilder(String page, String formName, String formId, String userInputName, 
                String passwordInputName, String submitButtonId, String submitButtonName)
        {
            this.page = page;
            this.formName = formName;
            this.formId = formId;
            this.userInputName = userInputName;
            this.passwordInputName = passwordInputName;
            this.submitButtonId = submitButtonId;
            this.submitButtonName = submitButtonName;
        }

        /**
         * Configure the log in function to look for a form with the name attribute of {@code formName}.  If the
         * builder was previously configured with a form ID, it will be removed to avoid confusion.
         * 
         * @param formName value of the name attribute to look for when identifying the form 
         * @return new builder configured to look for forms with {@code formName}
         */
        public LogInFunctionBuilder formName(String formName)
        {
            return new LogInFunctionBuilder(page, formName, null, userInputName, passwordInputName, submitButtonId, submitButtonName);
        }
        
        /**
         * Configure the log in function to look for a form with the id attribute of {@code formId}. If the
         * builder was previously configured with a form name, it will be removed to avoid confusion.
         * 
         * @param formId value of the id attribute to look for when identifying the form 
         * @return new builder configured to look for forms with {@code formId}
         */
        public LogInFunctionBuilder formId(String formId)
        {
            return new LogInFunctionBuilder(page, null, formId, userInputName, passwordInputName, submitButtonId, submitButtonName);
        }
        
        /**
         * Configure the log in function to look for an input field with the name attribute of {@code userInputName}.
         * 
         * @param userInputName value of the name attribute to look for when identifying the username input field 
         * @return new builder configured to look for username input fields with {@code userInputName}
         */
        public LogInFunctionBuilder userInputName(String userInputName)
        {
            return new LogInFunctionBuilder(page, formName, formId, userInputName, passwordInputName, submitButtonId, submitButtonName);
        }
        
        /**
         * Configure the log in function to look for an input field with the name attribute of {@code passwordInputName}.
         * 
         * @param passwordInputName value of the name attribute to look for when identifying the password input field 
         * @return new builder configured to look for password input fields with {@code passwordInputName}
         */
        public LogInFunctionBuilder passwordInputName(String passwordInputName)
        {
            return new LogInFunctionBuilder(page, formName, formId, userInputName, passwordInputName, submitButtonId, submitButtonName);
        }
        
        /**
         * Configure the log in function to look for a button with the id attribute of {@code submitButtonId}.  If the
         * builder was previously configured with a submit button name, it will be removed to avoid confusion.
         * 
         * @param submitButtonId value of the id attribute to look for when identifying the submit button 
         * @return new builder configured to look for a submit button with {@code submitButtonId}
         */
        public LogInFunctionBuilder submitButtonId(String submitButtonId)
        {
            return new LogInFunctionBuilder(page, formName, formId, userInputName, passwordInputName, submitButtonId, null);
        }
        
        /**
         * Configure the log in function to look for a button with the name attribute of {@code submitButtonName}.  If the
         * builder was previously configured with a submit button ID, it will be removed to avoid confusion.
         * 
         * @param submitButtonName value of the name attribute to look for when identifying the submit button 
         * @return new builder configured to look for a submit button with {@code submitButtonName}
         */
        public LogInFunctionBuilder submitButtonName(String submitButtonName)
        {
            return new LogInFunctionBuilder(page, formName, formId, userInputName, passwordInputName, null, submitButtonName);
        }

        /**
         * Build and return the final log in function.
         * @return final log in function
         */
        public Function<Login, HtmlPage> build()
        {
            return new GenericLogInFunction();
        }
        
        private final class GenericLogInFunction implements Function<Login, HtmlPage>
        {
            public HtmlPage apply(Login login)
            {
                final Doj loginPage = Doj.on(login.client.gotoPage(page));
                final Doj loginForm = getLogInForm(loginPage);
                final Doj os_username = loginForm.get("input").withName(userInputName);
                final Doj os_password = loginForm.get("input").withName(passwordInputName);
                final Doj submit = getSubmit(loginForm);

                if (!os_username.isEmpty() && !os_password.isEmpty() && !submit.isEmpty())
                {
                    os_username.value(login.username);
                    os_password.value(login.username);
                    try
                    {
                        return (HtmlPage) submit.click();
                    }
                    catch (IOException e)
                    {
                        throw new RuntimeException(e);
                    }
                }
                throw new IllegalStateException("No login form found in page " + loginPage);
            }

            private Doj getSubmit(final Doj loginForm)
            {
                final Doj submit;
                if (submitButtonId != null)
                {
                    submit = loginForm.getById(submitButtonId);
                }
                else if (submitButtonName != null)
                {
                    submit = loginForm.getByAttribute("name", submitButtonName);
                }
                else
                {
                    submit = loginForm.getByTag("input").withType("submit");
                }
                return submit;
            }

            private Doj getLogInForm(final Doj loginPage)
            {
                final Doj loginForm;
                if (formName != null)
                {
                    loginForm = loginPage.getByTag("form").withName(formName).first();
                }
                else if (formId != null)
                {
                    loginForm = loginPage.getById(formId).first();
                }
                else
                {
                    loginForm = loginPage.get("form").first();
                }
                return loginForm;
            }
        }
    }

    /**
     * Returns a function that determines if the page the {@code UiTester} is currently looking at is the log in page
     * by looking for a form with a name attribute of {@code formName}.
     * 
     * @param formName name of the form to look for
     * @return function that determines if the page the {@code UiTester} is currently looking at is the log in page by form name
     */
    public static Function<UiTester, Boolean> isOnLogInPageByFormName(String formName)
    {
        return new IsOnLogInPageByFormName(formName);
    }

    private static final class IsOnLogInPageByFormName implements Function<UiTester, Boolean>
    {
        private final String formName;

        public IsOnLogInPageByFormName(String formName)
        {
            this.formName = formName;
        }

        public Boolean apply(UiTester uiTester)
        {
            return !uiTester.currentPage().get("form").withName(formName).isEmpty();
        }
    }

    /**
     * Returns a function that determines if the page the {@code UiTester} is currently looking at is the log in page
     * by looking for a form with an id attribute of {@code formId}.
     * 
     * @param formId id of the form to look for
     * @return function that determines if the page the {@code UiTester} is currently looking at is the log in page by form id
     */
    public static Function<UiTester, Boolean> isOnLogInPageByFormId(String formId)
    {
        return new IsOnLogInPageByFormId(formId);
    }

    private static final class IsOnLogInPageByFormId implements Function<UiTester, Boolean>
    {
        private final String formId;

        public IsOnLogInPageByFormId(String formId)
        {
            this.formId = formId;
        }

        public Boolean apply(UiTester uiTester)
        {
            return !uiTester.currentPage().get("form").withId(formId).isEmpty();
        }
    }
    
    public static Function<Backup, Void> restoreUnsupported(String reason)
    {
        return new RestoreUnsupported(reason);
    }
    
    private final static class RestoreUnsupported implements Function<Backup, Void>
    {
        private final String reason;

        public RestoreUnsupported(String reason)
        {
            this.reason = reason;
        }

        public Void apply(Backup backup)
        {
            throw new UnsupportedOperationException(reason);
        }
    }
    
    public static final class BackupFile
    {
        public final File file;
        public final UiTester client;
        
        public BackupFile(File backup, UiTester client)
        {
            this.file = backup;
            this.client = client;
        }
        
        @Override
        public String toString()
        {
            return "Backup(" + file.toString() + ")";
        }
    }
    
    public static Function<Backup, Void> restore(Function<Backup, File> processBackupData, Function<BackupFile, Void> restore)
    {
        return new Restore(processBackupData, restore);
    }
    
    private static final class Restore implements Function<Backup, Void>
    {
        private final Function<Backup, File> processBackupData;
        private final Function<BackupFile, Void> restore;

        public Restore(Function<Backup, File> processBackupData, Function<BackupFile, Void> fn)
        {
            this.processBackupData = processBackupData;
            this.restore = fn;
        }

        public Void apply(Backup backup)
        {
            backup.client.logInAs(backup.username);
            File dataFile = null;
            try
            {
                dataFile = processBackupData.apply(backup);
                restore.apply(new BackupFile(dataFile, backup.client));
            }
            finally
            {
                backup.client.logout();
                if (dataFile != null)
                {
                    dataFile.delete();
                }
            }
            return null;
        }
        
    }

    public static String getHome()
    {
        if (isBlank(System.getProperty("homedir")))
        {
            throw new RestoreFromBackupException("System property for home directory - 'homedir' - not found. " +
                            "If you are using AMPS, make sure you are using at least version 3.3.  " +
                            "If you are running from your IDE or in some other environment, you'll need to " +
                            "figure out how to set the 'homedir' system property for that environment.");
        }
        return System.getProperty("homedir");
    }

    public static <A> A withJavascriptDisabled(UiTester uiTester, Functions.Function0<A> f)
    {
        // temporarily disable javascript because we are running into a bug in HtmlUnit
        // http://sourceforge.net/tracker/?func=detail&aid=3039471&group_id=47038&atid=448266
        boolean isJsEnabled = uiTester.isJavaScriptEnabled();
        uiTester.setJavaScriptEnabled(false);
        try
        {
            return f.apply();
        }
        finally
        {
            uiTester.setJavaScriptEnabled(isJsEnabled);
        }
    }

    /**
     * Executes a {@code Runnable} when logged in to the given {@code UiTester}
     * as the specified user.
     * 
     * @param uiTester the {@code UiTester}
     * @param user the user to log in as
     * @param runnable the {@code Runnable} to execute
     */
    public static void whenLoggedInAs(UiTester uiTester, String user, Runnable runnable)
    {
        uiTester.logInAs(user);
        try
        {
            runnable.run();
        }
        finally
        {
            uiTester.logout();
        }
    }
}
