package com.atlassian.integrationtesting.ui;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.atlassian.integrationtesting.ui.CompositeUiTester.Backup;
import com.atlassian.integrationtesting.ui.CompositeUiTester.Login;
import com.atlassian.integrationtesting.ui.CompositeUiTester.WebSudoLogin;
import com.atlassian.integrationtesting.ui.UiTesters.BackupFile;

import com.gargoylesoftware.htmlunit.html.HtmlPage;
import com.google.common.base.Function;

import org.apache.commons.io.output.ByteArrayOutputStream;

import be.roam.hue.doj.Doj;

import static com.atlassian.integrationtesting.ui.UiTesters.doNothingWebSudoLogin;
import static com.atlassian.integrationtesting.ui.UiTesters.getHome;
import static com.atlassian.integrationtesting.ui.UiTesters.goToPageToLogout;
import static com.atlassian.integrationtesting.ui.UiTesters.isOnLogInPageByFormName;
import static com.atlassian.integrationtesting.ui.UiTesters.logInByGoingTo;
import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.io.FileUtils.forceMkdir;
import static org.apache.commons.io.IOUtils.closeQuietly;
import static org.apache.commons.io.IOUtils.copy;
import static org.apache.commons.lang.RandomStringUtils.randomAlphanumeric;

/**
 * {@code UiTesterProvider}s for Confluence versions 3.1, 3.2, 3.3, and 3.4.
 */
public enum Confluence implements UiTesterFunctionProvider
{
    v3_1,
    v3_2,
    v3_3
    {
        @Override
        public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
        {
            return ConfluenceWebSudoLogin.INSTANCE;
        }
    },
    v3_4
    {
        @Override
        public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
        {
            return ConfluenceWebSudoLogin.INSTANCE;
        }
    },
    v3_5
    {
        @Override
        public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
        {
            return ConfluenceWebSudoLogin.INSTANCE;
        }
    };

    private final Function<UiTester, Void> logout = goToPageToLogout("logout.action");
    public Function<UiTester, Void> logout()
    {
        return logout;
    }

    public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
    {
        return doNothingWebSudoLogin();
    }

    private final Function<Login, HtmlPage> logIn = logInByGoingTo("login.action").formName("loginform").submitButtonId("loginButton").build();
    public Function<Login, HtmlPage> logIn()
    {
        return logIn;
    }

    private final Function<UiTester, Boolean> isOnLogInPage = isOnLogInPageByFormName("loginform");
    public Function<UiTester, Boolean> isOnLogInPage()
    {
        return isOnLogInPage;
    }

    public Function<UiTester, String> getLoggedInUser()
    {
        return UiTesters.getLoggedInUser();
    }

    private enum ConfluenceWebSudoLogin implements Function<CompositeUiTester.WebSudoLogin, HtmlPage>
    {
        INSTANCE;

        public HtmlPage apply(CompositeUiTester.WebSudoLogin login)
        {
            // Lets kick in the webSudo
            login.client.gotoPage("admin/viewgeneralconfig.action");

             final Doj form = login.client.currentPage().get("form").first();
             // We are on the web sudo login page
             if (form != null && form.attribute("action").endsWith("doauthenticate.action"))
             {
                 form.get("#password").value(login.password);
                 try
                 {
                     return (HtmlPage) form.get("input").withId("authenticateButton").click();
                 }
                 catch (IOException e)
                 {
                     throw new RuntimeException(e);
                 }
             }

             return (HtmlPage) login.client.currentPage().firstElement().getPage();
        }
    }

    
    public Function<Backup, Void> restore()
    {
        return UiTesters.restore(ProcessBackupData.INSTANCE, Restore.INSTANCE);
    }

    private enum Restore implements Function<BackupFile, Void>
    {
        INSTANCE;

        public Void apply(BackupFile backup)
        {
            try
            {
                backup.client.gotoPage("admin/backup.action");
                // select the backup file to restore
                backup.client.currentPage().getByTag("option").withValue(backup.file.getName()).select();
                // submit
                HtmlPage page = (HtmlPage) backup.client.currentPage().get("input").withType("submit").withValue("Restore").click();
                
                checkState(Doj.on(page).get("#title-text").text().contains("Importing Data - In Progress"));
                boolean complete = false;
                boolean failed = false;
                while (!complete && !failed)
                {
                    String importStatus = Doj.on(page).get("#taskCurrentStatus").text();
                    if ("Complete.".equals(importStatus))
                    {
                        complete = true;
                    }
                    else if (importStatus.startsWith("Import failed"))
                    {
                        failed = true;
                    }
                    else
                    {
                        try
                        {
                            Thread.sleep(500);
                        }
                        catch (InterruptedException e)
                        {
                            // ignore
                        }
                    }
                }
                if (failed)
                {
                    throw new RestoreFromBackupException("Failed to restore data. Check the Confluence logs for more information.");
                }
            }
            catch (ClassCastException e)
            {
                throw new RestoreFromBackupException("Element with id 'restore_submit' is not a button");
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException(e);
            }
            return null;
        }
    }
    
    private enum ProcessBackupData implements Function<Backup, File>
    {
        INSTANCE;
        
        private static final String ENTITIES_XML = "entities.xml";
        
        public File apply(Backup backup)
        {
            File dataFile = new File(getHome(), "restore/backup" + randomAlphanumeric(5) + ".zip");
            
            InputStream in = null;
            ZipInputStream zin = null;
            
            OutputStream out = null; 
            ZipOutputStream zout = null;
            
            try
            {
                forceMkdir(dataFile.getParentFile());
                dataFile.createNewFile();

                in = backup.data.openStream();
                zin = new ZipInputStream(in);
                
                out = new FileOutputStream(dataFile);
                zout = new ZipOutputStream(out);
                
                ZipEntry inEntry = null;
                while ((inEntry = zin.getNextEntry()) != null)
                {
                    if (inEntry.getName().equals(ENTITIES_XML))
                    {
                        ByteArrayOutputStream baos = new ByteArrayOutputStream();
                        copy(zin, baos);
                        String data = baos.toString().replaceAll("@base-url@", backup.client.getBaseUrl());

                        zout.putNextEntry(new ZipEntry(ENTITIES_XML));
                        copy(new StringReader(data), zout);
                        zout.closeEntry();
                    }
                    else
                    {
                        zout.putNextEntry(new ZipEntry(inEntry.getName()));
                        copy(zin, zout);
                        zout.closeEntry();
                    }
                }
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            finally
            {
                closeQuietly(zin);
                closeQuietly(in);
                closeQuietly(zout);
                closeQuietly(out);
            }
            return dataFile;
        }
    }
}
