package com.atlassian.integrationtesting.runner.restore;

import com.atlassian.integrationtesting.runner.CompositeTestRunner;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.AfterTestMethod;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.BeforeTestClass;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.BeforeTestMethod;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.Composer;
import com.atlassian.integrationtesting.ui.UiTester;

import com.google.common.base.Function;
import com.google.inject.Injector;

public final class RestoreFromBackup
{
    /**
     * Creates a {@code Composer} configured with functions to restore application data as per the specifications in
     * {@link Restore} and {@link RestoreOnce}.  The provided {@code Injector} is expected to be configured to provide
     * an instance of a {@link UiTester}, which handles the actual process of communicating with the application to
     * restore data.
     * 
     * @param injector Injector that can provide a {@link UiTester}
     * @return composer with functions for restoring backup data before class, before method and after methods are run 
     */
    public static Composer compose(Injector injector)
    {
        return CompositeTestRunner.compose().
            beforeTestClass(new BeforeClass(injector)).
            beforeTestMethod(new BeforeMethod(injector)).
            afterTestMethod(new AfterMethod(injector));
    }
    
    private static final class BeforeClass implements Function<BeforeTestClass, Void>
    {
        private final Injector injector;

        public BeforeClass(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(BeforeTestClass test)
        {
            if (test.hasAnnotation(RestoreOnce.class))
            {
                if (test.hasAnnotation(Restore.class))
                {
                    throw new RuntimeException("Both @Restore and @RestoreOnce found on class. Only one should be present.");
                }
                RestoreOnce restoreOnce = test.getAnnotation(RestoreOnce.class);
                injector.getInstance(UiTester.class).restore(restoreOnce.value(), restoreOnce.user());
            }
            return null;
        }
    }

    private static final class BeforeMethod implements Function<BeforeTestMethod, Void>
    {
        private final Injector injector;

        public BeforeMethod(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(BeforeTestMethod test)
        {
            if (test.hasAnnotation(Restore.class))
            {
                Restore restore = test.getAnnotation(Restore.class);
                injector.getInstance(UiTester.class).restore(restore.value(), restore.user());
            }
            return null;
        }
    }
    
    private static final class AfterMethod implements Function<AfterTestMethod, Void>
    {
        private final Injector injector;

        public AfterMethod(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(AfterTestMethod test)
        {
            if (test.hasAnnotation(Restore.class) && test.hasAnnotation(RestoreOnce.class))
            {
                injector.getInstance(UiTester.class).restore(test.getAnnotation(RestoreOnce.class).value());
            }
            return null;
        }
    }
}
