package com.atlassian.integrationtesting.runner;

import com.google.common.base.Predicate;

import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.TestClass;

import static org.apache.commons.lang.ArrayUtils.contains;

/**
 * <p>A runner that checks if tests in a class or individual test methods should be run based on the {@link TestGroups}
 * annotation.  If the class or method does not have a {@code TestGroups} annotation, the tests are run as normal.</p>
 * 
 * <p>If the {@code TestGroups} annotation is present, then {@link TestGroups#value()} and
 * {@link TestGroups#excludes()} are checked.  The tests in the class or test method will be ignored if
 * <ul>
 *   <li>The currently running test group is not in the list returned by {@code TestGroups#value()}</p>
 *   <li>Or if the currently running test group is in the list returned by {@link TestGroups#excludes()}
 * </ul>
 * Otherwise, the test will be run as normal.</p>
 * 
 * <p>The currently running test group is determined by checking for the system property named "testGroup".</p> 
 */
public final class TestGroupRunner extends CompositeTestRunner
{
    public TestGroupRunner(Class<?> klass) throws InitializationError
    {
        super(klass, compose());
    }

    /**
     * Returns a {@code Composer} that can be used to configure other runners with the same test predicates as
     * {@code TestGroupRunner}. 
     * 
     * @return a {@code Composer} that can be used to configure other runners with the same test predicates as
     * {@code TestGroupRunner} 
     */
    public static Composer compose()
    {
        return CompositeTestRunner.compose().
            shouldRunTestsInClass(shouldRunTestsInClass()).
            shouldRunTestMethod(shouldRunTestMethod());
    }

    private static Predicate<TestClass> shouldRunTestsInClass()
    {
        return ShouldRunTestsInClass.INSTANCE;
    }
    
    private enum ShouldRunTestsInClass implements Predicate<TestClass>
    {
        INSTANCE;

        public boolean apply(TestClass testClass)
        {
            return (shouldRun(testClass.getJavaClass().getAnnotation(TestGroups.class)));
        }
    }

    private static Predicate<FrameworkMethod> shouldRunTestMethod()
    {
        return ShouldRunTestMethod.INSTANCE;
    }
    
    private enum ShouldRunTestMethod implements Predicate<FrameworkMethod>
    {
        INSTANCE;

        public boolean apply(FrameworkMethod method)
        {
            return shouldRun(method.getAnnotation(TestGroups.class));
        }
    }

    private static boolean shouldRun(TestGroups testGroups)
    {
        if (testGroups == null)
        {
            return true;
        }

        String runningTestGroup = getRunningTestGroup();
        return includesTestGroup(testGroups, runningTestGroup) && !excludesTestGroup(testGroups, runningTestGroup);
    }

    private static boolean excludesTestGroup(TestGroups testGroups, String runningTestGroup)
    {
        return contains(testGroups.excludes(), runningTestGroup);
    }

    private static boolean includesTestGroup(TestGroups testGroups, String runningTestGroup)
    {
        return (testGroups.value().length == 1 && "*".equals(testGroups.value()[0]) || 
                contains(testGroups.value(), runningTestGroup));
    }

    /**
     * Return the id of the test group currently running.
     *  
     * @return id of the test group currently running
     */
    public static String getRunningTestGroup()
    {
        return System.getProperty("testGroup");
    }
}
