package com.atlassian.integrationtesting.runner;

import com.atlassian.integrationtesting.runner.CompositeTestRunner.AfterTestClass;
import com.atlassian.integrationtesting.runner.CompositeTestRunner.BeforeTestClass;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;

import org.junit.runner.notification.RunListener;
import org.junit.runner.notification.RunNotifier;

public final class RunListeners
{
    private RunListeners() {}
    
    /**
     * Creates a {@code Function} that will add the {@code listener} to the {@link RunNotifier} before the tests in a
     * class are run.
     *  
     * @param listener listener to be added
     * @return a {@code Function} that will add the {@code listener} to the {@link RunNotifier} before the tests in a
     * class are run
     */
    public static Function<BeforeTestClass, Void> addListener(RunListener listener)
    {
        return new AddListener(listener, Predicates.<BeforeTestClass>alwaysTrue());
    }
    
    /**
     * Creates a {@code Function} that will add the {@code listener} to the {@link RunNotifier} before the tests in a
     * class are run, but only if the {@code predicate} returns {@code true}.
     *  
     * @param listener listener to be added
     * @param predicate condition that must be {@code true} for the listener to be added 
     * @return a {@code Function} that will add the {@code listener} to the {@link RunNotifier} before the tests in a
     * class are run, but only if the {@code predicate} returns {@code true}.
     */
    public static Function<BeforeTestClass, Void> addListener(RunListener listener, Predicate<BeforeTestClass> predicate)
    {
        return new AddListener(listener, predicate);
    }
    
    private static final class AddListener implements Function<BeforeTestClass, Void>
    {
        private final RunListener listener;
        private final Predicate<BeforeTestClass> predicate;

        public AddListener(RunListener listener, Predicate<BeforeTestClass> predicate)
        {
            this.listener = listener;
            this.predicate = predicate;
        }

        public Void apply(BeforeTestClass test)
        {
            if (predicate.apply(test))
            {
                test.notifier.addListener(listener);
            }
            return null;
        }
    }

    /**
     * Creates a {@code Function} that will remove the {@code listener} from the {@link RunNotifier} after the tests
     * in a class are run.
     *  
     * @param listener listener to be removed
     * @return a {@code Function} that will remove the {@code listener} from the {@link RunNotifier} after the tests in
     * a class are run
     */
    public static Function<AfterTestClass, Void> removeListener(RunListener listener)
    {
        return new RemoveListener(listener);
    }
    
    private static final class RemoveListener implements Function<AfterTestClass, Void>
    {
        private final RunListener listener;

        public RemoveListener(RunListener listener)
        {
            this.listener = listener;
        }

        public Void apply(AfterTestClass test)
        {
            if (listener != null)
            {
                test.notifier.removeListener(listener);
            }
            return null;
        }
    }
}
