package com.atlassian.webdriver.integration.jira.element.internal;

import com.atlassian.pageobjects.PageBinder;
import com.atlassian.pageobjects.elements.ElementBy;
import com.atlassian.pageobjects.elements.PageElement;
import com.atlassian.pageobjects.elements.PageElementFinder;
import com.atlassian.pageobjects.elements.query.Conditions;
import com.atlassian.pageobjects.elements.query.Poller;
import com.atlassian.pageobjects.elements.timeout.TimeoutType;
import com.google.common.collect.Iterables;
import org.openqa.selenium.WebDriver;

import javax.annotation.Nonnull;
import javax.inject.Inject;

import static com.google.common.base.Predicates.equalTo;
import static com.google.common.base.Predicates.not;

/**
 * Page Object for OAuth confirmation page
 * This is stolen from Applinks Page Objects which are brittle. They expect login fields to be os_username and os_password.
 * We're using the Jira login page instead
 */
public class OAuthConfirmPage<T> {
    private final T nextPage;

    @Inject
    private WebDriver webDriver;

    @Inject
    private PageElementFinder elementFinder;

    @Inject
    private PageBinder pageBinder;

    @ElementBy(name = "approve")
    private PageElement allowField;

    public OAuthConfirmPage(T nextPage) {
        this.nextPage = nextPage;
    }

    public T confirmHandlingWebLoginIfRequired(String username, String password) {
        handleWebLoginIfRequiredThenClick(username, password, allowField);
        return nextPage;
    }

    public T confirmHandlingWebLoginIfRequired(String originalWindowHandle, String username, String password) {
        handleWebLoginIfRequiredThenClick(originalWindowHandle, username, password, allowField);
        return nextPage;
    }

    /**
     * Enters username and password into the login form if one is present and then click the element inside
     * the OAuth confirmation frame.
     *
     * @param username             username
     * @param password             password
     * @param fieldToClick         the element to click, null means do not click anything.
     */
    private void handleWebLoginIfRequiredThenClick(@Nonnull String username,
                                                   @Nonnull String password,
                                                   @Nonnull PageElement fieldToClick) {
        WebLoginPage maybeLoginPage = pageBinder.bind(WebLoginPage.class);
        Poller.waitUntilTrue(Conditions.or(fieldToClick.withTimeout(TimeoutType.SLOW_PAGE_LOAD).timed().isPresent(), maybeLoginPage.isShowing()));
        maybeLoginPage.handleWebLoginIfRequired(username, password);
        fieldToClick.click();
    }

    /**
     * Handles the web login (if required) and clicking the element inside the OAuth confirmation frame when the OAuth
     * flow opens in a new web browser window.
     *
     * @param originalWindowHandle the handle for the original window
     * @param username             username
     * @param password             password
     * @param fieldToClick         the element to click, null means do not click anything.
     */
    private void handleWebLoginIfRequiredThenClick(@Nonnull String originalWindowHandle,
                                                   @Nonnull String username,
                                                   @Nonnull String password,
                                                   @Nonnull PageElement fieldToClick) {
        String targetWindowHandle = Iterables.filter(webDriver.getWindowHandles(), not(equalTo(originalWindowHandle))).iterator().next();
        webDriver.switchTo().window(targetWindowHandle);
        handleWebLoginIfRequiredThenClick(username, password, fieldToClick);
        webDriver.switchTo().window(originalWindowHandle);
    }
}
