package com.atlassian.webdriver.integration.jira.element;

import com.atlassian.pageobjects.PageBinder;
import com.atlassian.pageobjects.elements.PageElement;
import com.atlassian.pageobjects.elements.query.Poller;
import com.atlassian.pageobjects.elements.query.TimedCondition;
import com.atlassian.pageobjects.elements.timeout.Timeouts;
import com.google.common.base.Function;
import org.apache.commons.lang.StringUtils;
import org.hamcrest.Description;
import org.hamcrest.TypeSafeMatcher;
import org.openqa.selenium.By;

import javax.inject.Inject;
import java.util.List;

public class JiraIssuePanel {

    private static final String ATTR_LAST_UPDATED = "data-last-updated";

    @Inject
    private PageBinder pageBinder;

    @Inject
    private Timeouts timeouts;

    private final PageElement container;
    private final JiraIssuesDialog parent;

    public JiraIssuePanel(PageElement container, JiraIssuesDialog parent) {
        this.container = container;
        this.parent = parent;
    }

    public String getAssignee() {
        return getComplexFieldText("assignee");
    }

    public String getIssueKey() {
        return getFieldText("key");
    }

    public String getDescription() {
        return getFieldText("description");
    }

    public String getIssueType() {
        return getComplexFieldText("type");
    }

    public String getPriority() {
        return getComplexFieldText("priority");
    }

    public String getStatus() {
        PageElement statusField = getField("status");
        // Jira 6.2+ is a lozenge. 6.1 and lower use the icon
        PageElement statusLozenge = statusField.find(By.className("jira-issue-status-lozenge"));
        PageElement statusIcon = statusField.find(By.tagName("img"));
        return statusLozenge.isPresent() ? statusLozenge.getText() :
                statusIcon.isPresent() ? statusIcon.getAttribute("alt").toUpperCase() : null;
    }

    public String getSummary() {
        return getFieldText("summary");
    }

    public JiraIssuePanel transitionIssue(String transitionName) {
        return executeTransition(transitionName, new Function<PageElement, JiraIssuePanel>() {
            @Override
            public JiraIssuePanel apply(final PageElement transition) {
                waitUntilUpdated(container, new Runnable() {
                    @Override
                    public void run() {
                        transition.click();
                    }
                });
                return parent.getIssuePanel();
            }
        });
    }

    public JiraTransitionForm transitionIssueExpectingForm(String transitionName) {
        return executeTransition(transitionName, new Function<PageElement, JiraTransitionForm>() {
            @Override
            public JiraTransitionForm apply(PageElement transition) {
                transition.click();
                JiraTransitionForm form = parent.getTransitionForm();
                Poller.waitUntilTrue(form.isShowing());
                return form;
            }
        });
    }

    private <T> T executeTransition(String transitionName, Function<PageElement, T> execute) {
        List<PageElement> transitions = container.findAll(By.className("jira-transition"));
        for (PageElement transition : transitions) {
            if (transitionName.equals(transition.getText())) {
                return execute.apply(transition);
            }
        }
        throw new IllegalArgumentException("Transition with name " + transitionName + " not found.");
    }

    private String getFieldText(String name) {
        return getField(name).getText();
    }

    private String getComplexFieldText(String name) {
        return getComplexField(name).getText();
    }

    private PageElement getField(String name) {
        return container.find(By.cssSelector(".issue-" + name));
    }

    private PageElement getComplexField(String name) {
        return container.find(By.cssSelector(".issue-" + name + " .field-text"));
    }

    public TimedCondition isShowing(String issueKey) {
        return getField("key").timed().hasText(issueKey);
    }

    private void waitUntilUpdated(PageElement element, Runnable operation) {
        Long lastUpdated = getLastUpdated(element);
        operation.run();
        waitUntilUpdated(element, lastUpdated);
    }

    private Long getLastUpdated(PageElement element) {
        return stringToLong(element.getAttribute(ATTR_LAST_UPDATED), null);
    }

    private void waitUntilUpdated(PageElement element, Long lastUpdated) {
        final long updated = lastUpdated == null ? -1L : lastUpdated;
        Poller.waitUntil(element.timed().getAttribute(ATTR_LAST_UPDATED), new TypeSafeMatcher<String>() {
            protected boolean matchesSafely(String value) {
                return stringToLong(value, -1L) > updated;
            }
            public void describeTo(Description description) {
                description.appendText("more recent than").appendValue(updated);
            }
        });
    }

    private Long stringToLong(String s, Long defaultValue) {
        if (StringUtils.isNotEmpty(s)) {
            try {
                return Long.parseLong(s);
            } catch (NumberFormatException e) {
                // no-op
            }
        }

        return defaultValue;
    }
}
