package com.atlassian.webdriver.integration.jira.element;

import com.atlassian.pageobjects.PageBinder;
import com.atlassian.pageobjects.binder.Init;
import com.atlassian.pageobjects.elements.PageElement;
import com.atlassian.pageobjects.elements.PageElementFinder;
import com.atlassian.pageobjects.elements.query.TimedCondition;
import com.atlassian.pageobjects.elements.timeout.TimeoutType;
import com.google.inject.Inject;
import org.openqa.selenium.By;

public class JiraForm {
    protected final PageElement container;
    @Inject
    protected PageElementFinder elementFinder;
    protected PageElement pageRoot;
    @Inject
    private PageBinder pageBinder;

    public JiraForm(PageElement container) {
        this.container = container;
    }

    public void clickSelectPageElement(String fieldId) {
        String id = "s2id_" + fieldId;
        elementFinder.find(By.id(id))
                .withTimeout(TimeoutType.SLOW_PAGE_LOAD)
                .click();

        // Attempt to try and make sure the dropdown list is present and the options are available.
        getAllowedValuesField(fieldId).waitUntilOptionsAvailable();
    }

    public void clearSelectPageElement(String fieldId) {
        String id = "s2id_" + fieldId;
        elementFinder.find(By.id(id))
                .find(By.className("select2-search-choice-close"))
                .withTimeout(TimeoutType.DIALOG_LOAD)
                .click();
    }

    public JiraTextField getTextField(String name) {
        return getField(JiraTextField.class, By.cssSelector("input[type=\"text\"][name=\"" + name + "\"]"));
    }

    public JiraTextareaField getTextareaField(String name) {
        return getField(JiraTextareaField.class, By.cssSelector("textarea[name=\"" + name + "\"]"));
    }

    public JiraAllowedValuesField getAllowedValuesField(String name) {
        return getField(JiraAllowedValuesField.class, By.cssSelector("select[name=\"" + name + "\"]"));
    }

    public JiraAutoCompleteField getAutoCompleteField(String name) {
        return getSelect2Field(JiraAutoCompleteField.class, By.cssSelector("div[data-jira-type=\"" + name + "\"]"),
                By.id("select2-drop"));
    }

    public JiraAssigneeField getAssigneeField(String name) {
        return getSelect2Field(JiraAssigneeField.class, By.cssSelector("div[data-jira-type=\"" + name + "\"]"),
                By.id("select2-drop"));
    }

    public JiraRadioButtonsField getRadioField(String type) {
        return getField(JiraRadioButtonsField.class, By.cssSelector("fieldset[data-jira-type=\"" + type + "\"]"));
    }

    public JiraCheckboxesField getCheckboxesField(String type) {
        return getField(JiraCheckboxesField.class, By.cssSelector("fieldset[data-jira-type=\"" + type + "\"]"));
    }

    public JiraTextField getLabelsField(String type) {
        return getField(JiraTextField.class, By.cssSelector("div[data-jira-type=\"" + type + "\"] input.select2-input"));
    }

    public TimedCondition isShowing() {
        return container.timed().isVisible();
    }

    private <T> T getField(Class<T> type, By locator) {
        scrollTo(locator);
        return pageBinder.bind(type, container.find(locator));
    }

    private <T> T getSelect2Field(Class<T> type, By locator, By hiddenInputLocator) {
        scrollTo(locator);
        return pageBinder.bind(type, container.find(locator), hiddenInputLocator, pageRoot);
    }

    private void scrollTo(By locator) {
        PageElement element = elementFinder.find(locator);
        scrollTo(element);
    }

    // We use the condition in the selectors because the plugin compat build runs against master that has data-testid="client-extensions-modal"
    // but the JIP runs again BBS 7.0.5 which has the data-testid="client-plugins-modal"
    // This code might break again if the CSE updates changes the modal data-testid
    // TODO: we should revert to use only one selector when the BBS version is >= 7.11
    protected void scrollTo(PageElement element){
        container.javascript().execute("AJS.$('[data-testid=client-plugins-modal],[data-testid=client-extensions-modal] header + div').scrollTop(" + element.getLocation().getY() + ")");
    }

    @Init
    private void initRoot() {
        if (pageRoot == null) {
            pageRoot = elementFinder.find(By.tagName("body"));
        }
    }
}
