package com.atlassian.webdriver.integration.jira.element;

import com.atlassian.pageobjects.elements.PageElement;
import com.atlassian.pageobjects.elements.query.Poller;
import com.atlassian.pageobjects.elements.query.webdriver.GenericWebDriverTimedQuery;
import org.openqa.selenium.By;

import java.util.List;

public class JiraAutoCompleteField implements JiraTransitionField {
    protected final PageElement selectField;
    protected By hiddenSelector;
    protected final PageElement pageRoot;
    private final PageElement container;

    public JiraAutoCompleteField(PageElement container, By hiddenSelector, PageElement pageRoot) {
        this.container = container;
        this.selectField = container.find(By.cssSelector(".select2-container > a"));
        this.hiddenSelector = hiddenSelector;
        this.pageRoot = pageRoot;
    }

    public PageElement getSelection() {
        return selectField;
    }

    @Override
    public String getValue() {
        return selectField.getValue();
    }

    @Override
    public void setValue(String val) {
        PageElement popup = performSearch(val);
        PageElement selectedItem = popup.find(By.cssSelector(".select2-highlighted"));
        selectedItem.click();
    }

    public void chooseValue(String val) {
        selectField.click();
        PageElement select2Dropdown = pageRoot.find(By.id("select2-drop"));
        Poller.waitUntilTrue(select2Dropdown.timed().isVisible());
        List<PageElement> options = select2Dropdown.findAll(By.cssSelector(".select2-results > li"));
        for (PageElement option : options) {
            if (option.getText().equals(val)) {
                option.click();
                break;
            }
        }
    }

    /**
     * This will wait until there is at least one option (apart from the placeholder) to select from
     * in the dropdown. Note that this is only effective when waiting for the initial population of a field,
     * subsequent updates won't always clear out the list of options before updating.
     *
     * @return the field for chaining
     */
    public JiraAutoCompleteField waitUntilOptionsAvailable() {
        scrollIntoView();
        Poller.waitUntilTrue(new GenericWebDriverTimedQuery<>(() -> {
            List<PageElement> options = container.findAll(By.cssSelector("select > option"));
            int size = options.size();
            if (size > 1) {
                return true;
            }
            if (size == 1) {
                String placeholder = container.find(By.cssSelector("select")).getAttribute("data-placeholder");
                if (placeholder == null) {
                    return true;
                }
                String optionText = options.get(0).getText();
                if (!placeholder.equals(optionText)) {
                    return true;
                }
            }
            // no options or the only option is a placeholder
            throw new GenericWebDriverTimedQuery.InvalidValue(false);
        }, 10000));
        return this;
    }

    protected PageElement performSearch(String val) {
        scrollIntoView();
        Poller.waitUntilTrue(selectField.timed().isVisible());
        selectField.click();
        PageElement popup = pageRoot.find(hiddenSelector);
        Poller.waitUntilTrue(popup.timed().isVisible());
        PageElement selectInput = popup.find(By.cssSelector("input"));

        selectInput.type(val);

        Poller.waitUntilFalse(selectInput.timed().hasClass("select2-active"));
        return popup;
    }

    protected void scrollIntoView() {
        selectField.javascript().execute("arguments[0].scrollIntoView();");
    }
}
