package com.atlassian.h2;

import io.atlassian.util.concurrent.LazyReference;
import org.h2.tools.Server;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.SQLException;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Encapsulates the lifecycle of a server instance.
 */
@ThreadSafe
public class ServerLifecycle {
    private final Supplier<Server> server;
    private final Supplier<ServerView> serverView = new LazyReference<ServerView>() {
        @Override
        protected ServerView create() {
            return new ServerView() {
                @Override
                public boolean isRunning() {
                    return server.get().isRunning(false);
                }

                @Override
                public URI getUri() {
                    try {
                        return new URI("jdbc:h2:tcp://localhost:" + server.get().getPort());
                    } catch (URISyntaxException e) {
                        throw new RuntimeException(e);
                    }
                }

                @Override
                public String toString() {
                    return server.toString();
                }
            };
        }
    };

    public ServerLifecycle(@Nonnull final Supplier<Server> serverFactory) {
        requireNonNull(serverFactory);
        this.server = new LazyReference<Server>() {
            @Override
            protected Server create() {
                return serverFactory.get();
            }
        };
    }

    @Nonnull
    public ServerView view() {
        return serverView.get();
    }

    @Nonnull
    public synchronized ServerView start() {
        final ServerView view = view();
        if (!view.isRunning()) {
            try {
                server.get().start();
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
        return view();
    }

    @Nonnull
    public synchronized ServerView stop() {
        final ServerView view = view();
        if (view.isRunning()) {
            server.get().stop();
        }
        return view();
    }
}
