package com.atlassian.h2;

import com.atlassian.config.ConfigurationException;
import com.atlassian.config.db.DatabaseDetails;
import org.h2.tools.Server;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.function.Supplier;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

/**
 * Encapsulates the lifecycle of a server instance, including automatically creating a database if none exists.
 * @since 2.0.0
 */
@ThreadSafe
public class DatabaseCreatingServerLifecycle extends ServerLifecycle {
    private static final String H2_JDBC_URL = "jdbc:h2:%s";
    private static final String H2_JDBC_URL_READ_ONLY = "jdbc:h2:%s;ACCESS_MODE_DATA=r";

    private final Supplier<File> databaseDirectory;
    private final String databaseName;

    public DatabaseCreatingServerLifecycle(@Nonnull Supplier<Server> serverFactory, @Nonnull Supplier<File> databaseDirectory, @Nonnull String databaseName) {
        super(serverFactory);
        this.databaseDirectory = requireNonNull(databaseDirectory);
        this.databaseName = requireNonNull(databaseName);
    }

    @Nonnull
    @Override
    public synchronized ServerView start() {
        ServerView view = view();
        if (!view.isRunning()) {

            // Connect to the DB using a mem connection to automatically create it.
            // Avoids needing to set the -ifNotExists server option to allow any TCP connection to create DBs
            String dbName = new File(databaseDirectory.get(), databaseName).getAbsolutePath();
            DatabaseDetails details;
            try {
                details = DatabaseDetails.getDefaults("h2");
            } catch (ConfigurationException e) {
                throw new RuntimeException("Cannot load database defaults", e);
            }
            validateH2DBIfExists(dbName, details);
            createH2IfNeeded(dbName, details);
            //Start H2 server once we are sure that the old file is compatible and created.
            view = super.start();
        }
        return view;
    }

    private void validateH2DBIfExists(String dbNameForURL, DatabaseDetails details) {
        String h2DBFileName = databaseName + ".mv.db";
        File databaseFile = new File(databaseDirectory.get(), h2DBFileName);
        if(databaseFile.exists()) {
            runSimpleSelectOnH2(format(H2_JDBC_URL_READ_ONLY, dbNameForURL), details);
        }
    }

    private void createH2IfNeeded(String dbNameForURL, DatabaseDetails details) {
        runSimpleSelectOnH2(format(H2_JDBC_URL, dbNameForURL), details);
    }

    private void runSimpleSelectOnH2(String jdbcURL, DatabaseDetails details) {
        try (
                Connection connection = DriverManager.getConnection(jdbcURL, details.getUserName(), details.getPassword());
                Statement statement = connection.createStatement();
                ResultSet resultSet = statement.executeQuery("select 1")) {
            resultSet.next();
        } catch (SQLException e) {
            throw new H2QueryFailedException(format("Cannot connect to %s.", databaseName), e);
        }
    }
}
