package com.atlassian.gadgets.dashboard;

import com.atlassian.annotations.PublicApi;
import com.google.common.base.Preconditions;
import io.atlassian.fugue.Option;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;
import static io.atlassian.fugue.Option.option;

/**
 * Detailed information that can be used to render the dashboard item.
 *
 * <p>
 * Dashboard items can take one of three forms:
 * <ul>
 * <li><b>Local JS dashboard item</b>: dashboard item rendered entirely in the browser. It specifies an AMD module which needs to be called in order to render the item.</li>
 * <li><b>Local server-side dashboard item</b>: dashboard item that has been rendered on the server. The representation contains an HTML which needs to be displayed. They may also contain AMD module for additional client-side processing.</li>
 * <li><b>Open-social gadget</b>: it contains url that can be used to render the gadget in an iFrame</li>
 * </ul>
 * </p>
 *
 * <p>
 * In case of local dashboard items, a {@code jsonRepresentation} will be returned which can be used to easily interact with the dashboard-plugin JavaScript API.
 * </p>
 */
@PublicApi
public class DashboardItemRepresentation {
    private final String jsonRepresentation;
    private final String html;
    private final String amdModule;
    private final String gadgetUrl;

    private DashboardItemRepresentation(@Nullable final String jsonRepresentation, @Nullable final String html, @Nullable final String amdModule, @Nullable final String gadgetUrl) {
        this.jsonRepresentation = jsonRepresentation;
        this.html = html;
        this.amdModule = amdModule;
        this.gadgetUrl = gadgetUrl;
    }

    /**
     * It is defined in case of local dashboard items and contains a stringified JSON object.
     *
     * <p>
     * Json representation contains the following fields:
     *
     * <ul>
     * <li>id</li>
     * <li>title</li>
     * <li>titleUrl</li>
     * <li>gadgetSpecUrl -- in case of local dashboard items that replace open social gadgets</li>
     * <li>height</li>
     * <li>width</li>
     * <li>color</li>
     * <li>column</li>
     * <li>colorUrl</li>
     * <li>gadgetUrl</li>
     * <li>isMaximizable</li>
     * <li>renderedGadgetUrl</li>
     * <li>hasNonHiddenUserPrefs</li>
     * <li>userPrefs</li>
     * <li>loaded</li>
     * <li>errorMessage</li>
     * <li>inlineHtml -- if it's a server-side local dashboard item</li>
     * <li>amdModule -- if it's a JavaScript local dashboard item</li>
     * <li>configurable</li>
     * <li>context</li>
     * </ul>
     * </p>
     *
     * @return a stringified JSON object with dashboard item rendering information.
     */
    public Option<String> getJsonRepresentation() {
        return option(jsonRepresentation);
    }

    /**
     * Returns a JavaScript AMD module that renders the dashboard item.
     *
     * @return amd module name
     */
    public Option<String> getAmdModule() {
        return option(amdModule);
    }

    /**
     * Returns a verbatim html which represents the dashboard item.
     *
     * @return unescaped html string
     */
    public Option<String> getHtml() {
        return option(html);
    }

    /**
     * Open social gadget URL
     *
     * @return url
     */
    public Option<String> getGadgetUrl() {
        return option(gadgetUrl);
    }

    public static DashboardItemRepresentation dashboardItem(@Nullable String amdModule, @Nullable String html, @Nonnull String jsonRepresentation) {
        Preconditions.checkArgument(amdModule != null || html != null);
        return new DashboardItemRepresentation(checkNotNull(jsonRepresentation), html, amdModule, null);
    }

    public static DashboardItemRepresentation openSocialGadget(@Nonnull String gadgetUrl) {
        return new DashboardItemRepresentation(null, null, null, checkNotNull(gadgetUrl));
    }
}
