package com.atlassian.gadgets.dashboard;

import com.atlassian.annotations.PublicApi;
import com.atlassian.gadgets.DashboardItemState;
import com.atlassian.gadgets.GadgetRequestContext;
import io.atlassian.fugue.Option;

/**
 * Service used to get representations of dashboard items.
 * <p>
 * It is useful when implementing custom dashboard views and using dashboard JavaScript API.
 * </p>
 *
 * @since v3.11
 */
@PublicApi
public interface DashboardItemRepresentationService {
    /**
     * Context in which the dashboard item is rendered.
     */
    final class RenderingContext {
        private final GadgetRequestContext requestContext;
        private final DashboardId dashboardId;
        private final DashboardState.ColumnIndex columnIndex;
        private final boolean writable;

        private RenderingContext(final GadgetRequestContext requestContext, final DashboardId dashboardId, final DashboardState.ColumnIndex columnIndex, final boolean writable) {
            this.requestContext = requestContext;
            this.dashboardId = dashboardId;
            this.columnIndex = columnIndex;
            this.writable = writable;
        }

        public static RenderingContext readOnly(final GadgetRequestContext requestContext, final DashboardId dashboardId, final DashboardState.ColumnIndex columnIndex) {
            return new RenderingContext(requestContext, dashboardId, columnIndex, false);
        }

        public static RenderingContext editable(final GadgetRequestContext requestContext, final DashboardId dashboardId, final DashboardState.ColumnIndex columnIndex) {
            return new RenderingContext(requestContext, dashboardId, columnIndex, true);
        }

        public GadgetRequestContext getRequestContext() {
            return requestContext;
        }

        public DashboardId getDashboardId() {
            return dashboardId;
        }

        public DashboardState.ColumnIndex getColumnIndex() {
            return columnIndex;
        }

        public boolean isWritable() {
            return writable;
        }
    }

    /**
     * Returns a dashboard item representation. For details of what the representation means look at the {@link DashboardItemRepresentation} documentation.
     *
     * @param itemState dashboard item state
     * @param context   rendering context
     * @return representation of the dashboard item or absent if the dashboard item cannot be rendered (conditions do not pass)
     */
    Option<DashboardItemRepresentation> getRepresentation(DashboardItemState itemState, RenderingContext context);
}
