package com.atlassian.gadgets;

import com.atlassian.gadgets.dashboard.Color;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.concurrent.Immutable;
import java.util.Collections;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Represents the state of the dashboard item which is not based on Open Social specification but is based on the description
 * provided in the plugin.
 * Extends {@link com.atlassian.gadgets.DashboardItemState} with:
 *
 * <ul>
 * <li>a complete key of the module</li>
 * <li>description</li>
 * <li>properties</li>
 * </ul>
 */
@Immutable
public final class LocalDashboardItemState implements DashboardItemState {
    private final GadgetId gadgetId;
    private final Color color;
    private final LocalDashboardItemModuleId dashboardItemModuleId;
    private final Map<String, String> properties;

    public LocalDashboardItemState(final GadgetId gadgetId, final Color color, final LocalDashboardItemModuleId dashboardItemModuleId, final Map<String, String> properties) {
        this.gadgetId = checkNotNull(gadgetId);
        this.color = checkNotNull(color);
        this.dashboardItemModuleId = dashboardItemModuleId;
        this.properties = ImmutableMap.copyOf(properties);
    }

    @Override
    public GadgetId getId() {
        return gadgetId;
    }

    @Override
    public Color getColor() {
        return color;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    public LocalDashboardItemModuleId getDashboardItemModuleId() {
        return dashboardItemModuleId;
    }

    @Override
    public <V> V accept(final DashboardItemStateVisitor<V> visitor) {
        return visitor.visit(this);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(final LocalDashboardItemState dashboardItemState) {
        return new Builder()
                .color(dashboardItemState.color)
                .gadgetId(dashboardItemState.gadgetId)
                .dashboardItemModuleId(dashboardItemState.dashboardItemModuleId)
                .properties(dashboardItemState.getProperties());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LocalDashboardItemState)) {
            return false;
        }
        LocalDashboardItemState rhs = (LocalDashboardItemState) o;
        return new EqualsBuilder()
                .append(getId(), rhs.getId())
                .append(getDashboardItemModuleId(), rhs.getDashboardItemModuleId())
                .append(getColor(), rhs.getColor())
                .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append(getId())
                .append(getDashboardItemModuleId())
                .append(getColor())
                .toHashCode();
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("id", getId())
                .append("dashboard-item-module-id", getDashboardItemModuleId())
                .append("color", getColor())
                .toString();
    }

    public static class Builder {
        private GadgetId gadgetId;
        private Color color = Color.defaultColor();
        private LocalDashboardItemModuleId dashboardItemModuleId;
        private Map<String, String> properties = Collections.emptyMap();

        public Builder gadgetId(final GadgetId gadgetId) {
            this.gadgetId = gadgetId;
            return this;
        }

        public Builder color(final Color color) {
            this.color = color;
            return this;
        }

        public Builder dashboardItemModuleId(final LocalDashboardItemModuleId dashboardItemModuleId) {
            this.dashboardItemModuleId = dashboardItemModuleId;
            return this;
        }

        public Builder properties(Map<String, String> properties) {
            this.properties = properties;
            return this;
        }

        public LocalDashboardItemState build() {
            return new LocalDashboardItemState(gadgetId, color, dashboardItemModuleId, properties);
        }
    }
}
