package com.atlassian.gadgets;

import io.atlassian.fugue.Option;
import com.atlassian.gadgets.view.View;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.util.Locale;
import java.util.function.Function;

/**
 * Holds information associated with a gadget request that the system will need to use.
 *
 * {@code GadgetRequestContext} objects should be created using the {@code GadgetRequestContext.Builder} class. By doing a
 * static import of the {@link GadgetRequestContext.Builder#gadgetRequestContext} method, you can create a {@code GadgetRequestContext} as follows:
 *
 * <pre>
 *  GadgetRequestContext gadgetRequestContext = gadgetRequestContext().user(new GadgetRequestContext.User("bob", "bob)).build();
 * </pre>
 */
public final class GadgetRequestContext {
    public static final class User {
        private final String userKey;
        private final String username;

        public User(String userKey, String username) {
            this.userKey = Preconditions.checkNotNull(userKey);
            this.username = Preconditions.checkNotNull(username);
        }

        public String getUserKey() {
            return userKey;
        }

        public String getUsername() {
            return username;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            User that = (User) o;

            return Objects.equal(this.userKey, that.userKey) &&
                    Objects.equal(this.username, that.username);
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(userKey, username);
        }

        @Override
        public String toString() {
            return String.format("User{userKey=%s, username=%s}", userKey, username);
        }
    }

    public static final GadgetRequestContext NO_CURRENT_REQUEST = Builder.gadgetRequestContext().locale(new Locale("")).ignoreCache(false).debug(false).build();

    private final Locale locale;
    private final boolean ignoreCache;
    private final boolean debug;
    private final Option<GadgetRequestContext.User> user;
    private final View view;

    private GadgetRequestContext(GadgetRequestContext.Builder builder) {
        this.locale = builder.locale;
        this.ignoreCache = builder.ignoreCache;
        this.view = builder.view;
        this.debug = builder.debug;
        this.user = builder.user;
    }

    /**
     * Returns the locale used for this request.
     *
     * @return the {@code Locale} used for this request
     */
    public Locale getLocale() {
        return locale;
    }

    /**
     * Returns the cache setting used for this request.
     *
     * @return the {@code RequestCacheSetting} used for this request
     */
    public boolean getIgnoreCache() {
        return ignoreCache;
    }

    /**
     * Returns the viewer (the current user) for this request.
     *
     * @return the username of this request viewer
     * @see com.atlassian.gadgets.GadgetRequestContext#getUser
     */
    public String getViewer() {
        return getUser().map(input -> input.getUsername()).getOrNull();
    }

    /**
     * Returns the dashboard view.
     *
     * @return dashboard view
     */
    public View getView() {
        return view;
    }

    /**
     * Returns {@code true} if the gadget should be rendered with debugging enabled, {@false} otherwise.  When this
     * returns {@code true}, the JavaScript for the features used by the gadget will be served in non-minimized, making
     * it easier to debug.
     *
     * @return {@code true} if the gadget should be rendered with debugging enabled, {@false} otherwise
     */
    public boolean isDebuggingEnabled() {
        return debug;
    }

    /**
     * Returns the current user or none if user is anonymous.
     *
     * @return current user
     */
    public Option<GadgetRequestContext.User> getUser() {
        return user;
    }

    /**
     * A builder that facilitates construction of {@code GadgetRequestContext} objects. The final {@code GadgetRequestContext}
     * is created by calling the {@link GadgetRequestContext.Builder#build()} method
     */
    public static class Builder {
        private Locale locale = Locale.US;
        private boolean ignoreCache = false;
        private View view = View.DEFAULT;
        private boolean debug = false;
        private Option<GadgetRequestContext.User> user = Option.none();

        /**
         * Get a new GadgetRequestContext Builder.
         *
         * @return the new builder
         */
        public static Builder gadgetRequestContext() {
            return new Builder();
        }

        /**
         * Returns the final constructed {@code GadgetRequestContext}.
         *
         * @return the {@code GadgetRequestContext}
         */
        public GadgetRequestContext build() {
            return new GadgetRequestContext(this);
        }

        /**
         * Set the {@code Locale} of the {@code GadgetRequestContext} under construction and return this {@code Builder}
         * to allow further construction to be done.
         *
         * @param locale the {@code Locale} to use for the {@code GadgetRequestContext}
         * @return this builder to allow for further construction
         */
        public Builder locale(Locale locale) {
            this.locale = locale;
            return this;
        }

        /**
         * Set the cache setting of the {@code GadgetRequestContext} under construction and return this {@code Builder}
         * to allow further construction to be done.
         *
         * @param ignoreCache the cache setting of this {@code GadgetRequestContext}
         * @return this builder to allow for further construction
         */
        public Builder ignoreCache(boolean ignoreCache) {
            this.ignoreCache = ignoreCache;
            return this;
        }

        /**
         * Set the current view of the {@code GadgetRequestContext} under construction and return this {@code Builder}
         * to allow further construction to be done.
         *
         * @param view view in which the dashboard item is rendering
         * @return this builder to allow for further construction
         */
        public Builder view(View view) {
            this.view = view;
            return this;
        }

        /**
         * Sets the current user.
         *
         * @param user the current user.
         * @return this builder to allow for further construction.
         */
        public Builder user(@Nullable GadgetRequestContext.User user) {
            this.user = Option.option(user);
            return this;
        }

        /**
         * Sets whether debugging should be enabled when rendering the gadget.  When debugging is enabled, the
         * JavaScript for the features used by the gadget will be served in non-minimized, making it easier to debug.
         *
         * @param debug enable or disable debugging
         * @return this builder to allow for further construction
         */
        public Builder debug(boolean debug) {
            this.debug = debug;
            return this;
        }
    }
}
