package com.atlassian.extras.decoder.api;

import com.atlassian.extras.common.LicenseException;

import java.util.Properties;

import static com.atlassian.extras.decoder.api.LicenseVerificationException.VerificationFailureReason.MISSING_PROPERTY;


public class LicenseVerificationException extends LicenseException {

    private static final String VERIFICATION_FAILURE_MSG_BASE = "Failed to verify license hash:";
    public static final String MISSING_PROPERTY_MSG = VERIFICATION_FAILURE_MSG_BASE + " missing property ";
    public static final String ERROR_DURING_VERIFICATION_MSG = VERIFICATION_FAILURE_MSG_BASE + " error during verification";
    public static final String VERIFICATION_FAILED_MSG = VERIFICATION_FAILURE_MSG_BASE + " verification failed";

    /**
     *  The reasons why the license signature verification may fail:
     *  {@link #MISSING_PROPERTY}
     *  {@link #ERROR_DURING_VERIFICATION}
     *  {@link #VERIFICATION_FAILED}
     */
    public enum VerificationFailureReason {
        /**
         * A license property mandatory for the signature verification is missing.
         * {@link #getAdditionalInfo} contains the missing property name.
         */
        MISSING_PROPERTY(MISSING_PROPERTY_MSG),
        /**
         * An error occured during the signature verification. See {@link #getMessage()} for more details.
         */
        ERROR_DURING_VERIFICATION(ERROR_DURING_VERIFICATION_MSG),
        /**
         * The signature is invalid.
         */
        VERIFICATION_FAILED(VERIFICATION_FAILED_MSG);

        public final String msg;

        VerificationFailureReason(String msg) {
            this.msg = msg;
        }
    }

    private final VerificationFailureReason reason;
    private final String additionalInfo;
    private final Properties properties;

    public LicenseVerificationException(VerificationFailureReason reason, Properties properties) {
        this(reason, null, properties);
    }

    public LicenseVerificationException(VerificationFailureReason reason, String additionalInfo, Properties properties) {
        this(reason, additionalInfo, properties, null);
    }

    public LicenseVerificationException(VerificationFailureReason reason, Properties properties, Throwable cause) {
        this(reason, null, properties, cause);
    }

    public LicenseVerificationException(VerificationFailureReason reason, String additionalInfo, Properties properties, Throwable cause) {
        super(reason.msg + (reason == MISSING_PROPERTY ? additionalInfo : ""), cause);
        this.reason = reason;
        this.additionalInfo = additionalInfo;
        this.properties = properties;
    }

    public VerificationFailureReason getReason() {
        return reason;
    }

    /**
     * @return may contain additional info related with the error
     */
    public String getAdditionalInfo() {
        return additionalInfo;
    }

    /**
     * @return the properties of the license that failed verification
     */
    public Properties getLicenseProperties() {
        return properties;
    }
}
