package com.atlassian.extras.core.transformer;

import com.atlassian.extras.api.Product;
import com.atlassian.extras.common.LicensePropertiesConstants;
import com.atlassian.extras.common.log.Logger;
import com.atlassian.extras.common.util.LicenseProperties;
import com.atlassian.extras.legacy.util.OldLicenseTypeResolver;
import com.atlassian.license.LicenseException;
import com.atlassian.license.LicenseTypeStore;

import java.util.Properties;

/**
 * Version 1 of the new licensing did not have LicenseEdition property.
 * Instead it carried the old license type name. Here we infer the right edition from the old name.
 */
public class LicenseEditionPropertyTransformer implements LicensePropertiesTransformer {
    private static final Logger.Log log = Logger.getInstance(LicenseEditionPropertyTransformer.class);

    public LicenseProperties transform(Product product, LicenseProperties properties) {
        String licenseTypeName = properties.getProperty(LicensePropertiesConstants.LICENSE_EDITION);
        if (licenseTypeName == null) {
            String oldLicenseTypeName = properties.getProperty(LicensePropertiesConstants.LICENSE_TYPE);
            if (oldLicenseTypeName != null) {
                LicenseTypeStore typeStore = OldLicenseTypeResolver.getLicenseTypeStore(product);
                if (typeStore != null) {
                    try {
                        com.atlassian.license.LicenseType oldType = typeStore.getLicenseType(oldLicenseTypeName);
                        if (oldType != null) {
                            Properties prop = new Properties();
                            prop.setProperty(LicensePropertiesConstants.LICENSE_EDITION, oldType.getEdition().name());

                            return new OverriddingLicenseProperties(product, properties, prop);
                        }
                    } catch (LicenseException e) {
                        log.warn("License type '" + oldLicenseTypeName + "' can not be resolved. This is a potentially corrupt license.", e);
                        return properties;
                    }
                }
            }
            log.warn("License type '" + oldLicenseTypeName + "' can not be resolved. This is a potentially corrupt license.");
            return properties;
        } else {
            // nothing to do
            return properties;
        }
    }

}
