package com.atlassian.extras.core;

import com.atlassian.extras.api.Contact;
import com.atlassian.extras.api.LicenseType;
import com.atlassian.extras.api.Organisation;
import com.atlassian.extras.api.Partner;
import com.atlassian.extras.api.Product;
import com.atlassian.extras.api.ProductLicense;
import com.atlassian.extras.common.LicenseTypeAndEditionResolver;
import com.atlassian.extras.common.util.LicenseProperties;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static com.atlassian.extras.common.LicensePropertiesConstants.ACTIVE_FLAG;
import static com.atlassian.extras.common.LicensePropertiesConstants.CONTACT_EMAIL;
import static com.atlassian.extras.common.LicensePropertiesConstants.CONTACT_NAME;
import static com.atlassian.extras.common.LicensePropertiesConstants.CREATION_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.DATA_CENTER;
import static com.atlassian.extras.common.LicensePropertiesConstants.DEFAULT_CREATION_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.DEFAULT_EXPIRY_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.DEFAULT_GRACE_PERIOD;
import static com.atlassian.extras.common.LicensePropertiesConstants.DEFAULT_MAX_USERS;
import static com.atlassian.extras.common.LicensePropertiesConstants.DESCRIPTION;
import static com.atlassian.extras.common.LicensePropertiesConstants.EVALUATION_LICENSE;
import static com.atlassian.extras.common.LicensePropertiesConstants.GRACE_PERIOD;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_EXPIRY_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_TYPE_NAME;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_VERSION;
import static com.atlassian.extras.common.LicensePropertiesConstants.MAINTENANCE_EXPIRY_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.MAX_NUMBER_OF_USERS;
import static com.atlassian.extras.common.LicensePropertiesConstants.NAMESPACE_SEPARATOR;
import static com.atlassian.extras.common.LicensePropertiesConstants.ORGANISATION;
import static com.atlassian.extras.common.LicensePropertiesConstants.PARTNER_NAME;
import static com.atlassian.extras.common.LicensePropertiesConstants.PURCHASE_DATE;
import static com.atlassian.extras.common.LicensePropertiesConstants.SERVER_ID;
import static com.atlassian.extras.common.LicensePropertiesConstants.SUBSCRIPTION_LICENSE;
import static com.atlassian.extras.common.LicensePropertiesConstants.SUPPORT_ENTITLEMENT_NUMBER;
import static com.atlassian.extras.common.LicensePropertiesConstants.UNLIMITED_USERS;
import static com.atlassian.extras.common.LicensePropertiesConstants.getKey;

/**
 * The default {@link ProductLicense} implementation.
 * <p>
 * Products <em>may</em> want to extend this class to add their product specific license attributes. However they
 * shouldn't need to override existing implementation.
 */
public class DefaultProductLicense implements ProductLicense {
    private final static long MILLIS_IN_A_DAY = 1000 * 60 * 60 * 24;

    private final int licenseVersion;
    private final String description;
    private final Product product;
    private final String serverId;
    private final Partner partner;
    private final Organisation organisation;
    private final Collection<Contact> contacts;
    private final Date creationDate;
    private final Date purchaseDate;
    private final int maximumNumberOfUsers;
    private final Date expiryDate;
    private final Date gracePeriodEndDate;
    private final Date maintenanceExpiryDate;
    private final String supportEntitlementNumber;
    private final boolean evaluation;
    private final boolean subscription;
    private final boolean clusteringEnabled;
    private final LicenseType licenseType;
    private final LicenseProperties properties;
    private final Iterable<Product> products;

    protected DefaultProductLicense(final Product product, final LicenseProperties properties) {
        this.licenseVersion = Integer.valueOf(properties.getProperty(LICENSE_VERSION, String.valueOf(0)));
        this.description = properties.getProperty(DESCRIPTION);
        this.product = product;
        this.evaluation = properties.getBoolean(EVALUATION_LICENSE);
        this.subscription = properties.getBoolean(SUBSCRIPTION_LICENSE);
        this.clusteringEnabled = properties.getBoolean(getKey(product, DATA_CENTER));
        this.serverId = properties.getProperty(SERVER_ID);
        this.partner = getPartner(properties);
        this.organisation = new DefaultOrganisation(properties.getProperty(ORGANISATION));
        this.contacts = getContacts(properties);
        this.creationDate = properties.getDate(CREATION_DATE, DEFAULT_CREATION_DATE);
        this.purchaseDate = properties.getDate(PURCHASE_DATE, creationDate); // default purchase date is creation date

        this.expiryDate = properties.getDate(LICENSE_EXPIRY_DATE, DEFAULT_EXPIRY_DATE);
        this.gracePeriodEndDate = getGracePeriodEndDate(properties, expiryDate);

        this.maintenanceExpiryDate = properties.getDate(MAINTENANCE_EXPIRY_DATE, DEFAULT_EXPIRY_DATE);
        this.supportEntitlementNumber = properties.getProperty(SUPPORT_ENTITLEMENT_NUMBER);
        this.maximumNumberOfUsers = properties.getInt(MAX_NUMBER_OF_USERS, DEFAULT_MAX_USERS);
        this.licenseType = LicenseTypeAndEditionResolver.getLicenseType(properties.getProperty(LICENSE_TYPE_NAME));
        this.properties = properties;

        //although a Product gets passed in as a parameter, each license can contain multiple embedded products as well.
        List<Product> productList = new ArrayList<Product>();
        for (String property : properties.getPropertiesEndingWith(NAMESPACE_SEPARATOR + ACTIVE_FLAG).keySet()) {
            productList.add(
                    Product.fromNamespace(property.substring(0, property.indexOf(NAMESPACE_SEPARATOR + ACTIVE_FLAG))));
        }
        this.products = productList;
    }

    private Date getGracePeriodEndDate(LicenseProperties properties, Date expiryDate) {
        if (expiryDate == null) {
            return null;
        }
        final int gracePeriod = properties.getInt(GRACE_PERIOD, DEFAULT_GRACE_PERIOD);
        return new Date(expiryDate.getTime() + (MILLIS_IN_A_DAY * gracePeriod));
    }

    public Product getProduct() {
        return product;
    }

    public String getServerId() {
        return serverId;
    }

    public Partner getPartner() {
        return partner;
    }

    public Organisation getOrganisation() {
        return organisation;
    }

    public Collection<Contact> getContacts() {
        return contacts;
    }

    public Date getCreationDate() {
        return new Date(creationDate.getTime());
    }

    public Date getPurchaseDate() {
        return new Date(purchaseDate.getTime());
    }

    public Date getExpiryDate() {
        return expiryDate != null ? new Date(expiryDate.getTime()) : null;
    }

    public int getNumberOfDaysBeforeExpiry() {
        if (expiryDate == null) {
            return Integer.MAX_VALUE;
        }
        return getDaysBeforeDate(expiryDate);
    }

    public boolean isExpired() {
        return expiryDate != null && expiryDate.compareTo(new Date()) < 0;
    }

    public Date getGracePeriodEndDate() {
        return gracePeriodEndDate != null ? new Date(gracePeriodEndDate.getTime()) : null;
    }

    public int getNumberOfDaysBeforeGracePeriodExpiry() {
        if (gracePeriodEndDate == null) {
            return Integer.MAX_VALUE;
        }
        return getDaysBeforeDate(gracePeriodEndDate);
    }

    public boolean isWithinGracePeriod() {
        return isExpired() && !isGracePeriodExpired();
    }

    public boolean isGracePeriodExpired() {
        return gracePeriodEndDate != null && gracePeriodEndDate.compareTo(new Date()) < 0;
    }

    public Date getMaintenanceExpiryDate() {
        return maintenanceExpiryDate != null ? new Date(maintenanceExpiryDate.getTime()) : null;
    }

    public int getNumberOfDaysBeforeMaintenanceExpiry() {
        if (maintenanceExpiryDate == null) {
            return Integer.MAX_VALUE;
        }

        return getDaysBeforeDate(maintenanceExpiryDate);
    }

    public boolean isMaintenanceExpired() {
        return maintenanceExpiryDate != null && maintenanceExpiryDate.compareTo(new Date()) < 0;
    }

    public String getSupportEntitlementNumber() {
        return supportEntitlementNumber;
    }

    public int getMaximumNumberOfUsers() {
        return maximumNumberOfUsers;
    }

    public boolean isUnlimitedNumberOfUsers() {
        return maximumNumberOfUsers == UNLIMITED_USERS;
    }

    public boolean isEvaluation() {
        return evaluation;
    }

    public boolean isSubscription() {
        return subscription;
    }

    public boolean isClusteringEnabled() {
        return clusteringEnabled;
    }

    public String getProperty(String name) {
        return properties.getProperty(name);
    }

    public Iterable<Product> getProducts() {
        return products;
    }

    private int getDaysBeforeDate(final Date date) {
        return (int) ((date.getTime() - System.currentTimeMillis()) / MILLIS_IN_A_DAY);
    }

    private static Partner getPartner(LicenseProperties properties) {
        final String partnerName = properties.getProperty(PARTNER_NAME);
        return partnerName != null ? new DefaultPartner(partnerName) : null;
    }

    private static Collection<Contact> getContacts(LicenseProperties properties) {
        final String contactEmail = properties.getProperty(CONTACT_EMAIL);
        final String contactName = properties.getProperty(CONTACT_NAME);
        if (contactEmail != null || contactName != null) {
            return Collections.<Contact>singletonList(new DefaultContact(contactName, contactEmail));
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * A simple implementation of the {@link Partner} interface.
     */
    private final static class DefaultPartner implements Partner {
        private final String name;

        DefaultPartner(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    /**
     * A simple implementation of the {@link Organisation} interface.
     */
    private final static class DefaultOrganisation implements Organisation {
        private final String name;

        DefaultOrganisation(final String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    /**
     * A simple implementation of the {@link Contact} interface.
     */
    private final static class DefaultContact implements Contact {
        private final String name;
        private final String email;

        DefaultContact(final String name, final String email) {
            this.name = name;
            this.email = email;
        }

        public String getName() {
            return name;
        }

        public String getEmail() {
            return email;
        }
    }

    public int getLicenseVersion() {
        return licenseVersion;
    }

    public String getDescription() {
        return description;
    }

    public LicenseType getLicenseType() {
        return licenseType;
    }
}
