package com.atlassian.extras.core;

import com.atlassian.extras.api.AtlassianLicense;
import com.atlassian.extras.api.LicenseManager;
import com.atlassian.extras.common.LicenseException;
import com.atlassian.extras.decoder.api.LicenseDecoder;

import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * A default implementation of the license manager, that
 */
final public class DefaultLicenseManager implements LicenseManager {
    private final Map<String, AtlassianLicense> licenses = new ConcurrentHashMap<String, AtlassianLicense>(1);

    private final LicenseDecoder licenseDecoder;

    private final AtlassianLicenseFactory atlassianLicenseFactory;

    public DefaultLicenseManager(final LicenseDecoder licenseDecoder, final AtlassianLicenseFactory atlassianLicenseFactory) {
        if (licenseDecoder == null) {
            throw new IllegalArgumentException("licenseDecoder must NOT be null!");
        }
        if (atlassianLicenseFactory == null) {
            throw new IllegalArgumentException("atlassianLicenseFactory must NOT be null!");
        }
        this.atlassianLicenseFactory = atlassianLicenseFactory;
        this.licenseDecoder = licenseDecoder;
    }

    public AtlassianLicense getLicense(String licenseString) {
        if (licenseString == null) {
            throw new IllegalArgumentException("licenseString must NOT be null");
        }

        try {
            return getAtlassianLicense(licenseString);
        } catch (LicenseException e) {
            throw e;
        } catch (RuntimeException t) {
            throw new LicenseException(t);
        }
    }

    private AtlassianLicense getAtlassianLicense(String licenseString) {
        AtlassianLicense license = licenses.get(licenseString);
        if (license == null) {
            license = decodeLicense(licenseString);
            if (license == null) {
                // this shouldn't happen but hey one never knows ;-)
                throw new LicenseException("Could not decode license <" + licenseString + ">, " +
                        "decoding returned a null Atlassian license object");
            }
            licenses.put(licenseString, license);
        }
        return license;
    }

    private AtlassianLicense decodeLicense(String licenseString) {
        return atlassianLicenseFactory.getLicense(licenseDecoder.decode(licenseString));
    }

    // following accessors exist for testing purpose

    Map<String, AtlassianLicense> getLicenses() {
        return Collections.unmodifiableMap(licenses);
    }

    LicenseDecoder getLicenseDecoder() {
        return licenseDecoder;
    }

    AtlassianLicenseFactory getAtlassianLicenseFactory() {
        return atlassianLicenseFactory;
    }

    public void clear() {
        licenses.clear();
    }
}
