package com.atlassian.extras.core;

import com.atlassian.extras.api.AtlassianLicense;
import com.atlassian.extras.api.Contact;
import com.atlassian.extras.api.LicenseType;
import com.atlassian.extras.api.Organisation;
import com.atlassian.extras.api.Partner;
import com.atlassian.extras.api.Product;
import com.atlassian.extras.api.ProductLicense;
import com.atlassian.extras.common.util.LicenseProperties;
import com.atlassian.extras.common.util.ProductLicenseProperties;
import com.atlassian.extras.core.plugins.DefaultPluginLicense;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import static com.atlassian.extras.api.Product.ALL_PLUGINS;
import static com.atlassian.extras.common.LicensePropertiesConstants.LICENSE_TYPE_NAME;

class DefaultAtlassianLicense implements AtlassianLicense {
    private final Map<Product, ProductLicense> productLicenseMap;

    DefaultAtlassianLicense(Collection<? extends ProductLicense> productLicenses) {
        productLicenseMap = new HashMap<Product, ProductLicense>(productLicenses.size());
        for (ProductLicense license : productLicenses) {
            productLicenseMap.put(license.getProduct(), license);
        }
    }

    /**
     * @see com.atlassian.extras.api.AtlassianLicense#getProductLicenses()
     * @since 2.2.0
     */
    public Collection<ProductLicense> getProductLicenses() {
        return Collections.unmodifiableCollection(productLicenseMap.values());
    }

    public ProductLicense getProductLicense(final Product product) {
        ProductLicense license = productLicenseMap.get(product);
        if (license != null) {
            return license;
        }
        if (product.isPlugin() && productLicenseMap.containsKey(ALL_PLUGINS)) {
            final ProductLicense delegate = productLicenseMap.get(ALL_PLUGINS);
            return new DelegatingPluginLicense(product, delegate);
        }
        return null;
    }

    private static final class DelegatingPluginLicense extends DefaultPluginLicense {
        private final ProductLicense delegate;

        private DelegatingPluginLicense(Product product, ProductLicense delegate) {
            super(product, newLicenseProperties(product, delegate));
            this.delegate = delegate;
        }

        private static LicenseProperties newLicenseProperties(Product product, ProductLicense license) {
            Properties props = new Properties();
            props.setProperty(LICENSE_TYPE_NAME, license.getProperty(LICENSE_TYPE_NAME));
            return new ProductLicenseProperties(product, props);
        }

        @Override
        public int getLicenseVersion() {
            return delegate.getLicenseVersion();
        }

        @Override
        public String getDescription() {
            return delegate.getDescription();
        }

        @Override
        public String getServerId() {
            return delegate.getServerId();
        }

        @Override
        public Partner getPartner() {
            return delegate.getPartner();
        }

        @Override
        public Organisation getOrganisation() {
            return delegate.getOrganisation();
        }

        @Override
        public Collection<Contact> getContacts() {
            return delegate.getContacts();
        }

        @Override
        public Date getCreationDate() {
            return delegate.getCreationDate();
        }

        @Override
        public Date getPurchaseDate() {
            return delegate.getPurchaseDate();
        }

        @Override
        public Date getExpiryDate() {
            return delegate.getExpiryDate();
        }

        @Override
        public int getNumberOfDaysBeforeExpiry() {
            return delegate.getNumberOfDaysBeforeExpiry();
        }

        @Override
        public boolean isExpired() {
            return delegate.isExpired();
        }

        @Override
        public Date getGracePeriodEndDate() {
            return delegate.getGracePeriodEndDate();
        }

        @Override
        public int getNumberOfDaysBeforeGracePeriodExpiry() {
            return delegate.getNumberOfDaysBeforeGracePeriodExpiry();
        }

        @Override
        public boolean isWithinGracePeriod() {
            return delegate.isWithinGracePeriod();
        }

        @Override
        public boolean isGracePeriodExpired() {
            return delegate.isGracePeriodExpired();
        }

        @Override
        public String getSupportEntitlementNumber() {
            return delegate.getSupportEntitlementNumber();
        }

        @Override
        public Date getMaintenanceExpiryDate() {
            return delegate.getMaintenanceExpiryDate();
        }

        @Override
        public int getNumberOfDaysBeforeMaintenanceExpiry() {
            return delegate.getNumberOfDaysBeforeMaintenanceExpiry();
        }

        @Override
        public boolean isMaintenanceExpired() {
            return delegate.isMaintenanceExpired();
        }

        @Override
        public int getMaximumNumberOfUsers() {
            return delegate.getMaximumNumberOfUsers();
        }

        @Override
        public boolean isUnlimitedNumberOfUsers() {
            return delegate.isUnlimitedNumberOfUsers();
        }

        @Override
        public boolean isEvaluation() {
            return delegate.isEvaluation();
        }

        @Override
        public boolean isSubscription() {
            return delegate.isSubscription();
        }

        @Override
        public boolean isClusteringEnabled() {
            return delegate.isClusteringEnabled();
        }

        @Override
        public LicenseType getLicenseType() {
            return delegate.getLicenseType();
        }

        @Override
        public String getProperty(String name) {
            return delegate.getProperty(name);
        }
    }
}
