/**
 *
 */
package com.atlassian.extras.common.util;

import com.atlassian.extras.api.Product;
import com.atlassian.extras.common.DateEditor;
import com.atlassian.extras.common.LicensePropertiesConstants;

import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Implementation of {@link LicenseProperties} that will look up the property in the {@link Product} namespace first and
 * fall back to global properties if not found.
 */
public class ProductLicenseProperties implements LicenseProperties {
    private final Product product;
    private final Properties properties;

    public ProductLicenseProperties(final Product product, final Properties properties) {
        this.product = product;
        if (product == null) {
            throw new IllegalArgumentException("Product must NOT be null!");
        }
        if (properties == null) {
            throw new IllegalArgumentException("Properties must NOT be null!");
        }

        this.properties = properties;
    }

    public String getProperty(String s) {
        return getProperty(s, null);
    }

    public String getProperty(String s, String defaultValue) {
        final String o = properties.getProperty(LicensePropertiesConstants.getKey(product, s));
        return o != null ? o : properties.getProperty(s, defaultValue);
    }

    public int getInt(String propertyName, int defaultValue) {
        final String stringValue = getProperty(propertyName);
        if (stringValue == null || stringValue.length() == 0) {
            return defaultValue;
        }

        try {
            return Integer.parseInt(stringValue);
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public Date getDate(final String key, final Date defaultValue) {
        final String stringValue = getProperty(key);
        if (stringValue == null || stringValue.length() == 0) {
            return defaultValue;
        } else {
            return DateEditor.getDate(stringValue);
        }
    }

    public boolean getBoolean(final String key) {
        return Boolean.valueOf(getProperty(key));
    }

    public Map<String, String> getPropertiesEndingWith(final String ending) {
        Map<String, String> props = new HashMap<String, String>();
        for (Enumeration enumeration = properties.propertyNames(); enumeration.hasMoreElements(); ) {
            String propName = enumeration.nextElement().toString();
            if (propName.endsWith(ending)) {
                props.put(propName, getProperty(propName));
            }
        }
        return props;
    }
}