package com.atlassian.extras.common;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class used to manipulate dates in licenses.
 * The time zone used for all dates is Sydney/Australia.
 */
public final class DateEditor {
    private static final String DATE_FORMAT = "yyyy-MM-dd";

    static final String PERIOD_PREFIX = "P";

    private static final long MILLIS_IN_HOUR = 60 * 60 * 1000;

    public static final String UNLIMITED = "unlimited";

    private static final Pattern DURATION_PATTERN = Pattern.compile("Duration\\:([0-9]+)");

    private static final Pattern PERIOD_PATTERN = Pattern.compile(PERIOD_PREFIX + "([0-9]+)H");

    private static final Pattern DATE_IN_MILLIS_PATTERN = Pattern.compile("[0-9]+");

    private static final Pattern ISO_DATE_PATTERN = Pattern.compile("^([1-2][0-9]{3}-(0[1-9]|1[0-2])-([0-2][0-9]|3[0-1]))(\\s|[T])?.*");

    static final TimeZone TIME_ZONE = TimeZone.getTimeZone("Australia/Sydney");


    /**
     * Calculate the date that the {@code dateString} represents.
     * That is:
     * <ol>
     * <li>{@link #UNLIMITED} represents a date <em>far away in the future</em> and {@code null} will be returned,
     * <li>The end of the period when the string starts with {@link #PERIOD_PREFIX},</li>
     * <li>The end of the period when the string starts with "Duration",</li>
     * <li>Expressed in milliseconds from the epoc time,</li>
     * <li>Else the date in ISO format</li>
     * </ol>
     *
     * @param dateString the date represented as a string
     * @return the correct date, or {@code null} to represent a date <em>far away in the future</em>.
     * @see #getString(Date)
     */
    public static Date getDate(String dateString) {
        if (dateString == null || dateString.length() == 0) {
            throw new DateParsingException(dateString);
        }

        // unlimited
        if (dateString.equals(UNLIMITED)) {
            return null; //
        }

        // Duration
        final Matcher durationMatcher = DURATION_PATTERN.matcher(dateString);
        if (durationMatcher.matches()) {
            final long dateInMillis = System.currentTimeMillis() + Long.parseLong(durationMatcher.group(1));
            return new Date(dateInMillis);
        }

        // Period
        final Matcher periodMatcher = PERIOD_PATTERN.matcher(dateString);
        if (periodMatcher.matches()) {
            final long dateInMillis = System.currentTimeMillis()
                    + Integer.parseInt(periodMatcher.group(1)) * MILLIS_IN_HOUR;

            return new Date(dateInMillis);
        }

        // Date in millis
        final Matcher dateInMillisMatcher = DATE_IN_MILLIS_PATTERN.matcher(dateString);
        if (dateInMillisMatcher.matches()) {
            return new Date(Long.parseLong(dateString));
        }

        // "normal" date
        final Matcher isoDateMatcher = ISO_DATE_PATTERN.matcher(dateString);
        if (isoDateMatcher.matches()) {
            try {
                return getDateFormat().parse(isoDateMatcher.group(1));
            } catch (ParseException e) ///CLOVER:OFF
            {
                // should never happen!
                throw new DateParsingException(dateString, e);
            }
        }

        throw new DateParsingException(dateString);
    }

    /**
     * Returns the String representation of a {@link Date date}.
     *
     * @param date the given date
     * @return the ISO format of the date, {@link #UNLIMITED} if the given date is {@code null}
     */
    public static String getString(Date date) {
        return date != null ? getDateFormat().format(date) : UNLIMITED;
    }

    private static DateFormat getDateFormat() {
        final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
        dateFormat.setTimeZone(TIME_ZONE);
        return dateFormat;
    }
}
