package com.atlassian.extras.api;

import java.util.Collection;
import java.util.Date;

/**
 * Defines the license for a single instance of a single product.
 */
public interface ProductLicense {
    /**
     * The version of the license (i.e. which encoder was used to encode the license String), 0 if none set.
     */
    int getLicenseVersion();

    /**
     * User friendly description of this license
     */
    String getDescription();

    /**
     * Returns the product to which this license is associated to.
     * <p>
     * If additional licenses are embedded, please use {@link #getProducts()}. In this case, this method will
     * return just the host application {@link Product} but not the embedded ones.
     *
     * @return the product to which this license is associated to.
     * @see #getProducts()
     */
    Product getProduct();

    /**
     * Returns all {@link Product}s which are associated with this license. This can be any number of {@link Product}s
     * due to being able to embed additional products (e.g. OnDemand plugins) into a host license.
     *
     * @return all {@link Product}s which are associated with this license.
     * @see #getProduct()
     * @since 3.1.1
     */
    Iterable<Product> getProducts();

    /**
     * @return the server id associated with this license.
     */
    String getServerId();

    /**
     * @return the partner associated to this license, {@code null} if none.
     */
    Partner getPartner();

    /**
     * @return the organisation that <em>bought</em> the license.
     */
    Organisation getOrganisation();

    /**
     * @return the list of contacts in the {@link Organisation} for this license.
     */
    Collection<Contact> getContacts();

    /**
     * @return the date the license was created
     */
    Date getCreationDate();

    /**
     * @return the date the license was purchased.
     */
    Date getPurchaseDate();

    /**
     * @return the license expiry date, {@code null} if the license never expires.
     */
    Date getExpiryDate();

    /**
     * Returns the number of days before license expires. This method should not be used to determine whether the
     * license has expired as it is only accurate down to a day. Use {@link #isExpired()} instead.
     *
     * @return the number of days before expiry, {@link Integer#MAX_VALUE} if the license never expires.
     * @see #isExpired()
     * @see #getNumberOfDaysBeforeMaintenanceExpiry()
     */
    int getNumberOfDaysBeforeExpiry();

    /**
     * @return {@code true} if the license is expired, {@code false} otherwise.
     * @see #getExpiryDate()
     */
    boolean isExpired();

    /**
     * <p>Returns the expiry date of the grace period. The grace period extends from the {@link #getExpiryDate() expiry
     * date} to the date returned by this method. This period is typically used to alert customers that their license
     * has expired but they will still work fully for some period of time.</p>
     * <p>If the license doesn't expire then this method will return {@code null}, just as {@link #getExpiryDate()}.</p>
     *
     * @return the date at which the grace period ends, {@code null} if the license never expires.
     * @see #getExpiryDate()
     */
    Date getGracePeriodEndDate();

    /**
     * <p>Returns the number of day before the grace period expires. This method should not be used to determine whether the
     * grace period has ended as it is only accurate down to a day. Use {#isGracePeriodExpired()} instead.</p>
     * <p>If the license doesn't expire (no need for grace period expiry) it will return {@link Integer#MAX_VALUE}.</p>
     * <p>This method will return a valid number of days otherwise, even if the license hasn't expired yet and the grace
     * period still hasn't been activated. Users should use {#isWithinGracePeriod()} to check whether the grace period
     * has been activated</p>
     *
     * @return the number of days before the grace period expires.
     */
    int getNumberOfDaysBeforeGracePeriodExpiry();

    /**
     * Tells whether {@link System#currentTimeMillis() now} is after the {@link #getExpiryDate() expiry date} and
     * before the {@link #getGracePeriodEndDate() grace period end date}. This method will always return {@code false}
     * for licenses that don't expire.
     *
     * @return {@code true} if now is between the two dates mentionned, {@code false} otherwise.
     * @see #getGracePeriodEndDate()
     * @see #getExpiryDate()
     */
    boolean isWithinGracePeriod();

    /**
     * @return {@code true} if the license is past the {@link #getGracePeriodEndDate() grace period end date}, {@code false}
     * if it is {@code null} or the license is not past the grace period end date.
     * @see #getGracePeriodEndDate()
     */
    boolean isGracePeriodExpired();

    /**
     * @return the Support Entitlement Number (SEN), {@code null} if it cannot be found in the license.
     */
    String getSupportEntitlementNumber();

    /**
     * @return the expiry date for the maintenance on this license, {@code null} if the maintenance never expires.
     */
    Date getMaintenanceExpiryDate();

    /**
     * Returns the number of days before the maintenance expires. This method should not be used to determine whether
     * the maintenance has expired or not as it is only accurate down to a day. Use {@link #isMaintenanceExpired()}
     * instead.
     *
     * @return the number of days before maintenance expiry, {@link Integer#MAX_VALUE} if the maintenance never
     * expires.
     * @see #isMaintenanceExpired()
     * @see #getNumberOfDaysBeforeExpiry()
     */
    int getNumberOfDaysBeforeMaintenanceExpiry();

    /**
     * @return {@code true} if the maintenance for this license is expired, {@code false} otherwise.
     * @see #getMaintenanceExpiryDate()
     */
    boolean isMaintenanceExpired();

    /**
     * @return the maximum number of users allowed for this product instance, -1 if unlimited.
     * @see #isUnlimitedNumberOfUsers()
     */
    int getMaximumNumberOfUsers();

    /**
     * @return {@code true} if the license is for unlimited number of users, {@code false} otherwise.
     * @see #getMaximumNumberOfUsers()
     */
    boolean isUnlimitedNumberOfUsers();

    /**
     * @return {@code true} if this is an evaluation license, {@code false} otherwise.
     */
    boolean isEvaluation();

    /**
     * @return {@code true} if this is a subscription license, {@code false} otherwise.
     */
    boolean isSubscription();

    /**
     * @return {@code true} if this is a clustering license, {@code false} otherwise.
     * @since 3.2
     */
    public boolean isClusteringEnabled();

    /**
     * @return the license type.
     */
    LicenseType getLicenseType();

    /**
     * @param name the name of the property to look for
     * @return the property from the license, {@code null} if the property doesn't exist.
     */
    String getProperty(String name);
}
