package com.atlassian.extras.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Representation of products that Atlassian sells and generates licenses for customers to use.
 * <p>
 * <strong>NOTE:</strong> the case of the product name is not important with respect to object equality.
 */
public final class Product {
    private static final String JIRA_NS = "jira";
    private static final String CONF_NS = "conf";
    private static final String BAMBOO_NS = "bamboo";
    private static final String CROWD_NS = "crowd";
    private static final String CLOVER_NS = "clover";
    private static final String FISHEYE_NS = "fisheye";
    private static final String CRUCIBLE_NS = "crucible";
    private static final String STASH_NS = "stash";
    private static final String EDIT_LIVE_PLUGIN_NS = "edit_live_plugin";
    private static final String PERFORCE_PLUGIN_NS = "perforce_plugin";
    private static final String SHAREPOINT_PLUGIN_NS = "sharepoint_plugin";
    private static final String GREENHOPPER_NS = "greenhopper";
    private static final String TEAM_CALENDARS_NS = "team_calendars";
    private static final String BONFIRE_NS = "bonfire";
    private static final String VSS_PLUGIN_NS = "vss_plugin";
    private static final String CONF_QUESTIONS_NS = "conf_questions";

    public static Product JIRA = new Product("JIRA", JIRA_NS);
    public static Product CONFLUENCE = new Product("Confluence", CONF_NS);
    public static Product BAMBOO = new Product("Bamboo", BAMBOO_NS);
    public static Product CROWD = new Product("Crowd", CROWD_NS);
    public static Product CLOVER = new Product("Clover", CLOVER_NS);
    public static Product FISHEYE = new Product("FishEye", FISHEYE_NS);
    public static Product CRUCIBLE = new Product("Crucible", CRUCIBLE_NS);
    public static Product STASH = new Product("Stash", STASH_NS);
    public static Product BITBUCKET_SERVER = new Product("Bitbucket Server", STASH_NS);
    public static Product EDIT_LIVE_PLUGIN = new Product("Edit Live Plugin", EDIT_LIVE_PLUGIN_NS, true);
    public static Product PERFORCE_PLUGIN = new Product("Perforce Plugin", PERFORCE_PLUGIN_NS, true);
    public static Product SHAREPOINT_PLUGIN = new Product("Sharepoint Plugin", SHAREPOINT_PLUGIN_NS, true);
    public static Product GREENHOPPER = new Product("GreenHopper", GREENHOPPER_NS, true);
    public static Product TEAM_CALENDARS = new Product("Team Calendars", TEAM_CALENDARS_NS, true);
    public static Product BONFIRE = new Product("Bonfire", BONFIRE_NS, true);
    public static Product VSS_PLUGIN = new Product("VSS Plugin", VSS_PLUGIN_NS, true);
    public static Product CONFLUENCE_QUESTIONS = new Product("Confluence Questions", CONF_QUESTIONS_NS);

    public static Iterable<Product> ATLASSIAN_PRODUCTS = Collections.unmodifiableList(Arrays.asList(
            JIRA,
            CONFLUENCE,
            BAMBOO,
            CROWD,
            CLOVER,
            FISHEYE,
            CRUCIBLE,
            STASH,
            BITBUCKET_SERVER,
            EDIT_LIVE_PLUGIN,
            PERFORCE_PLUGIN,
            SHAREPOINT_PLUGIN,
            GREENHOPPER,
            TEAM_CALENDARS,
            BONFIRE,
            VSS_PLUGIN,
            CONFLUENCE_QUESTIONS
    ));

    public static Product[] getAtalssianProductsAsArray() {
        List<Product> products = new ArrayList<Product>();
        for (Product p : Product.ATLASSIAN_PRODUCTS) {
            products.add(p);
        }
        return products.toArray(new Product[]{});
    }

    /**
     * A special product that really denotes any plugin.  If this product is enabled in the license, then all plugins
     * will be enabled.
     */
    public static Product ALL_PLUGINS = new Product("All Plugins", "plugins.*", true);

    private final String name;
    private final String namespace;
    private final boolean plugin;

    public Product(String name, String namespace) {
        this(name, namespace, false);
    }

    public Product(String name, String namespace, boolean plugin) {
        this.name = name;
        this.plugin = plugin;

        // As of extras 3.0, we decided that all plugins
        // would have a namespace of their plugin key.
        // However, this alters the parameters that are encoded
        // into each license, meaning that they are no longer
        // properly read after being decoded by older extras libraries
        // (even though they are compatible with extras 3.0).
        // As a result, for the following plugin keys, let's
        // revert back to using their legacy namespaces.
        if ("com.atlassian.bonfire.plugin".equals(namespace)) {
            this.namespace = BONFIRE_NS;
        } else if ("com.atlassian.confluence.extra.team-calendars".equals(namespace)) {
            this.namespace = TEAM_CALENDARS_NS;
        } else if ("com.pyxis.greenhopper.jira".equals(namespace)) {
            this.namespace = GREENHOPPER_NS;
        } else if ("com.atlassian.confluence.extra.sharepoint".equals(namespace)) {
            this.namespace = SHAREPOINT_PLUGIN_NS;
        } else if ("com.atlassian.jira.plugin.ext.vss".equals(namespace)) {
            this.namespace = VSS_PLUGIN_NS;
        } else {
            this.namespace = namespace;
        }
    }

    /**
     * @return the name of the product, this will NEVER return {@code null}
     */
    public String getName() {
        return name;
    }

    /**
     * Namespace is used in licensing to group parameters that belong to this particular product only.
     *
     * @return product namespace
     */
    public String getNamespace() {
        return namespace;
    }

    public boolean isPlugin() {
        return plugin;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Product p = (Product) o;
        if (name != null ? !name.equalsIgnoreCase(p.name) : p.name != null) return false;
        return true;
    }

    @Override
    public int hashCode() {
        return 31 * 17 + name.hashCode();
    }

    @Override
    public String toString() {
        return "product <" + getName() + ">";
    }

    /**
     * Performs a {@link Product} lookup by namespace. If no such {@link Product} is found, a new one is created.
     *
     * @param ns namespace
     * @return the {@link Product} associated with the namespace
     * @since 3.1.1
     */
    public static Product fromNamespace(String ns) {
        if (JIRA_NS.equals(ns)) return JIRA;
        if (CONF_NS.equals(ns)) return CONFLUENCE;
        if (BAMBOO_NS.equals(ns)) return BAMBOO;
        if (CROWD_NS.equals(ns)) return CROWD;
        if (CLOVER_NS.equals(ns)) return CLOVER;
        if (FISHEYE_NS.equals(ns)) return FISHEYE;
        if (CRUCIBLE_NS.equals(ns)) return CRUCIBLE;
        if (STASH_NS.equals(ns)) return STASH;
        if (EDIT_LIVE_PLUGIN_NS.equals(ns)) return EDIT_LIVE_PLUGIN;
        if (PERFORCE_PLUGIN_NS.equals(ns)) return PERFORCE_PLUGIN;
        if (SHAREPOINT_PLUGIN_NS.equals(ns)) return SHAREPOINT_PLUGIN;
        if (GREENHOPPER_NS.equals(ns)) return GREENHOPPER;
        if (TEAM_CALENDARS_NS.equals(ns)) return TEAM_CALENDARS;
        if (BONFIRE_NS.equals(ns)) return BONFIRE;
        if (VSS_PLUGIN_NS.equals(ns)) return VSS_PLUGIN;
        if (CONF_QUESTIONS_NS.equals(ns)) return CONFLUENCE_QUESTIONS;

        //If it doesn't already exist, it must be a plugin (third parameter should be "true").
        return new Product(ns, ns, true);
    }
}