package com.atlassian.event.internal;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.event.config.ListenerHandlersConfiguration;
import com.atlassian.event.spi.EventDispatcher;
import com.atlassian.event.spi.ListenerHandler;
import com.atlassian.event.spi.ListenerInvoker;
import com.atlassian.plugin.eventlistener.descriptors.EventListenerModuleDescriptor;
import com.atlassian.plugin.scope.EverythingIsActiveScopeManager;
import com.atlassian.plugin.scope.ScopeManager;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static org.apache.commons.lang.ObjectUtils.identityToString;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

/**
 * <p>The default implementation of the {@link com.atlassian.event.api.EventPublisher} interface.</p>
 * <p>
 * <p>One can customise the event listening by instantiating with custom
 * {@link com.atlassian.event.spi.ListenerHandler listener handlers} and the event dispatching through
 * {@link com.atlassian.event.spi.EventDispatcher}. See the {@link com.atlassian.event.spi} package
 * for more information.</p>
 *
 * @see com.atlassian.event.spi.ListenerHandler
 * @see com.atlassian.event.spi.EventDispatcher
 * @since 2.0
 */
public final class EventPublisherImpl implements EventPublisher {
    private static final Logger log = LoggerFactory.getLogger(EventPublisherImpl.class);

    private static final String PROPERTY_PREFIX = EventPublisherImpl.class.getName();
    private static final Optional<String> debugRegistration =
            Optional.ofNullable(System.getProperty(PROPERTY_PREFIX + ".debugRegistration"));
    private static final boolean debugRegistrationLocation =
            Boolean.getBoolean(PROPERTY_PREFIX + ".debugRegistrationLocation");
    private static final Optional<String> debugInvocation =
            Optional.ofNullable(System.getProperty(PROPERTY_PREFIX + ".debugInvocation"));
    private static final boolean debugInvocationLocation =
            Boolean.getBoolean(PROPERTY_PREFIX + ".debugInvocationLocation");

    private final EventDispatcher eventDispatcher;
    private final List<ListenerHandler> listenerHandlers;
    private final ScopeManager scopeManager;

    /**
     * <strong>Note:</strong> this field makes this implementation stateful
     */
    private final Multimap<Class<?>, KeyedListenerInvoker> listenerInvokers;

    /**
     * <p>If you need to customise the asynchronous handling, you should use the
     * {@link com.atlassian.event.internal.AsynchronousAbleEventDispatcher} together with a custom executor. You might
     * also want to have a look at using the {@link com.atlassian.event.internal.EventThreadFactory} to keep the naming
     * of event threads consistent with the default naming of the Atlassian Event library.<p>
     *
     * @param eventDispatcher               the event dispatcher to be used with the publisher
     * @param listenerHandlersConfiguration the list of listener handlers to be used with this publisher
     * @see com.atlassian.event.internal.AsynchronousAbleEventDispatcher
     * @see com.atlassian.event.internal.EventThreadFactory
     */
    public EventPublisherImpl(EventDispatcher eventDispatcher, ListenerHandlersConfiguration listenerHandlersConfiguration) {
        this(eventDispatcher, listenerHandlersConfiguration, new EverythingIsActiveScopeManager());
    }

    /**
     * Inherits {@link EventPublisherImpl#EventPublisherImpl(EventDispatcher, ListenerHandlersConfiguration)} and
     * allows injection of scope manager
     *
     * @param eventDispatcher               the event dispatcher to be used with the publisher
     * @param listenerHandlersConfiguration the list of listener handlers to be used with this publisher
     * @param scopeManager                  the scope manager
     * @see com.atlassian.event.internal.AsynchronousAbleEventDispatcher
     * @see com.atlassian.event.internal.EventThreadFactory
     * @see ScopeManager
     */
    public EventPublisherImpl(EventDispatcher eventDispatcher,
                              ListenerHandlersConfiguration listenerHandlersConfiguration,
                              ScopeManager scopeManager) {
        this.eventDispatcher = checkNotNull(eventDispatcher);
        this.listenerHandlers = checkNotNull(checkNotNull(listenerHandlersConfiguration).getListenerHandlers());
        this.listenerInvokers = newMultimap();
        this.scopeManager = checkNotNull(scopeManager);
    }

    /**
     * The order in which registered listeners are invoked is predictable. Listeners will be invoked for listeners registered
     * on the object itself, then listeners on the parent class, then the grandparent and so on until finally all listeners for java.lang.Object are invoked.
     * After walking the class hierarchy the interface listeners are invoked, again from the most specific interface first.  Note that the ordering within a specific
     * event type is not guaranteed.  If there are multiple registered listeners for IssueEvent, then they will be invoked in the order of registration.
     * It is however guaranteed that a listener for IssueEvent will be invoked before a listener for Event
     *
     * @param event the event to publish
     */
    public void publish(Object event) {
        invokeListeners(findListenerInvokersForEvent(checkNotNull(event)), event);
    }

    public void register(final Object listener) {
        registerListener(identityToString(checkNotNull(listener)), listener);
    }

    public void unregister(Object listener) {
        unregisterListener(identityToString(checkNotNull(listener)));
    }

    public void unregisterAll() {
        synchronized (listenerInvokers) {
            listenerInvokers.clear();
        }
    }

    private void unregisterListener(String listenerKey) {
        checkArgument(isNotEmpty(listenerKey), "Key for the listener must not be empty");

        /** see {@link Multimaps#synchronizedMultimap(Multimap)} for why this synchronize block is there */
        synchronized (listenerInvokers) {
            for (Iterator<Map.Entry<Class<?>, KeyedListenerInvoker>> invokerIterator = listenerInvokers.entries().iterator(); invokerIterator.hasNext(); ) {
                if (invokerIterator.next().getValue().getKey().equals(listenerKey)) {
                    invokerIterator.remove();
                }
            }
        }
    }

    private void registerListener(String listenerKey, Object listener) {
        final Object listenerImpl;
        final Optional<String> listenerScope;
        if (listener instanceof EventListenerModuleDescriptor) {
            final EventListenerModuleDescriptor descriptor = (EventListenerModuleDescriptor) listener;
            listenerImpl = descriptor.getModule();
            listenerScope = descriptor.getScopeKey();
        } else {
            listenerImpl = listener;
            listenerScope = Optional.empty();
        }

        synchronized (listenerInvokers) /* Because we need to un-register an re-register in one 'atomic' operation */ {
            unregisterListener(listenerKey);

            final List<ListenerInvoker> invokers = Lists.newArrayList();
            for (ListenerHandler listenerHandler : listenerHandlers) {
                invokers.addAll(listenerHandler.getInvokers(listenerImpl));
            }
            if (!invokers.isEmpty()) {
                registerListenerInvokers(listenerKey, listenerScope, invokers);
            } else {
                throw new IllegalArgumentException("No listener invokers were found for listener <" + listenerImpl + ">");
            }
        }
    }

    private Set<KeyedListenerInvoker> findListenerInvokersForEvent(Object event) {
        final Set<KeyedListenerInvoker> invokers = newLinkedHashSet();
        /** see {@link Multimaps#synchronizedMultimap(Multimap)} for why this synchronize block is there */
        synchronized (listenerInvokers) {
            for (Class<?> eventClass : ClassUtils.findAllTypes(checkNotNull(event).getClass())) {
                invokers.addAll(listenerInvokers.get(eventClass));
            }
        }

        final List<KeyedListenerInvoker> activeInvokers = invokers
                .stream()
                .filter(i -> i.getScope()
                        .map(s -> scopeManager.isScopeActive(s))
                        .orElse(true))
                .collect(Collectors.toList());

        return newLinkedHashSet(activeInvokers);
    }

    private void invokeListeners(Collection<KeyedListenerInvoker> listenerInvokers, Object event) {
        final String eventClass = event.getClass().getName();
        final boolean debugThisInvocation = debugInvocation.map(eventClass::startsWith).orElse(false);
        for (KeyedListenerInvoker keyedInvoker : listenerInvokers) {
            final ListenerInvoker invoker = keyedInvoker.getInvoker();
            if (debugThisInvocation) {
                log.warn("Listener invoked event with class '{}' -> invoker {}", eventClass, invoker);
                if (debugInvocationLocation) {
                    log.warn("Invoked from", new Exception());
                }
            }
            // EVENT-14 -  we should continue to process all listeners even if one throws some horrible exception
            try {
                eventDispatcher.dispatch(invoker, event);
            } catch (Exception e) {
                log.error("There was an exception thrown trying to dispatch event '{}' from the invoker '{}'.",
                        event, invoker, e);
            }
        }
    }

    private void registerListenerInvokers(String listenerKey, Optional<String> scope, List<? extends ListenerInvoker> invokers) {
        for (ListenerInvoker invoker : invokers) {
            registerListenerInvoker(listenerKey, scope, invoker);
        }
    }

    private void registerListenerInvoker(String listenerKey, Optional<String> scope, ListenerInvoker invoker) {
        // if supported classes is empty, then all events are supported.
        if (invoker.getSupportedEventTypes().isEmpty()) {
            putEventListenerInvoker(Object.class, listenerKey, scope, invoker);
        }

        // if it it empty, we won't loop, otherwise register the invoker against all its classes
        for (Class<?> eventClass : invoker.getSupportedEventTypes()) {
            putEventListenerInvoker(eventClass, listenerKey, scope, invoker);
        }
    }

    private void putEventListenerInvoker(final Class<?> eventClass, String listenerKey, Optional<String> scope, ListenerInvoker invoker) {
        debugRegistration.ifPresent(classPrefix -> {
            if (eventClass.getName().startsWith(classPrefix)) {
                log.warn("Listener registered event '{}' -> invoker {}", eventClass, invoker);
                if (debugRegistrationLocation) {
                    log.warn("Registered from", new Exception());
                }
            }
        });

        listenerInvokers.put(eventClass, new KeyedListenerInvoker(listenerKey, invoker, scope));
    }

    private Multimap<Class<?>, KeyedListenerInvoker> newMultimap() {
        return Multimaps.synchronizedMultimap(
                Multimaps.newMultimap(Maps.newHashMap(), () -> Sets.newHashSet()));
    }

    private static final class KeyedListenerInvoker {
        private final String key;
        private final ListenerInvoker invoker;
        private final Optional<String> scope;

        KeyedListenerInvoker(String key, ListenerInvoker invoker, Optional<String> scope) {
            this.invoker = invoker;
            this.key = key;
            this.scope = scope;
        }

        String getKey() {
            return key;
        }

        ListenerInvoker getInvoker() {
            return invoker;
        }

        Optional<String> getScope() {
            return scope;
        }

        @Override
        public int hashCode() {
            return new HashCodeBuilder(5, 23).append(key).append(invoker).toHashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj == null || obj.getClass() != getClass()) {
                return false;
            }
            final KeyedListenerInvoker kli = (KeyedListenerInvoker) obj;
            return new EqualsBuilder()
                    .append(key, kli.key)
                    .append(invoker, kli.invoker)
                    .append(scope, kli.scope)
                    .isEquals();
        }
    }
}
