package com.atlassian.event;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;

import java.util.*;

/**
 * Manages the firing and receiving of events.
 *
 * <p>Any event passed to {@link #publishEvent} will be passed through Spring's event
 * model, after which it will be returned to the EventManager for processing via
 * {@link #onApplicationEvent}.
 *
 * <p>Plugin-based event listeners can register to receive events via
 * {@link #registerListener(String, com.atlassian.event.EventListener)}
 */
public class DefaultEventManager implements EventManager, ApplicationListener, ApplicationContextAware
{
    private static final Logger log = Logger.getLogger(DefaultEventManager.class);

    private ApplicationContext applicationContext;
    private Map listenersByKey = new HashMap();
    private Map listenersByClass = new HashMap();
    private List globalListeners = new ArrayList();

    public void setApplicationContext(ApplicationContext applicationContext)
    {
        this.applicationContext = applicationContext;
    }

    public void publishEvent(Event event)
    {
        if (event == null)
        {
            log.warn("Null event fired", new Exception("Null event fired"));
            return;
        }

        applicationContext.publishEvent(event);
    }

    public void onApplicationEvent(ApplicationEvent event)
    {
        if (event instanceof Event)
        {
            Event confluenceEvent = (Event) event;
            sendEventTo(confluenceEvent, globalListeners);
            sendEventTo(confluenceEvent, calculateListeners(confluenceEvent.getClass()));
        }
        else if (log.isDebugEnabled())
            log.debug("Ignoring non-Confluence event: " + event.getClass().getName() + " - " + event.toString());
    }

    private Collection calculateListeners(Class eventClass)
    {
        Set listeners = new HashSet();
        calculateListeners(eventClass, listeners);
        return listeners;
    }

    private void calculateListeners(Class eventClass, Set listeners)
    {
        if (eventClass == null || eventClass.equals(Event.class) || eventClass.equals(Object.class))
            return;

        final Collection matchingListeners = (Collection) listenersByClass.get(eventClass);

        if (matchingListeners != null)
            listeners.addAll(matchingListeners);

        for (int i = 0; i < eventClass.getInterfaces().length; i++)
        {
            Class aClass = eventClass.getInterfaces()[i];
            calculateListeners(aClass, listeners);
        }

        calculateListeners(eventClass.getSuperclass(), listeners);
    }

    public void registerListener(String listenerKey, com.atlassian.event.EventListener listener)
    {
        if (StringUtils.isEmpty(listenerKey))
            throw new IllegalArgumentException("Key for the listener must not be null: " + listenerKey);

        if (listener == null)
            throw new IllegalArgumentException("The listener must not be null: " + listener);

        if (listenersByKey.containsKey(listenerKey))
            unregisterListener(listenerKey);

        Class[] classes = listener.getHandledEventClasses();

        if (classes.length == 0)
            globalListeners.add(listener);

        for (int i = 0; i < classes.length; i++)
            addToListenerList(classes[i], listener);

        listenersByKey.put(listenerKey, listener);
    }

    public void unregisterListener(String listenerKey)
    {
        com.atlassian.event.EventListener listener = (com.atlassian.event.EventListener) listenersByKey.get(listenerKey);

        for (Iterator it = listenersByClass.values().iterator(); it.hasNext();)
        {
            List list = (List) it.next();
            list.remove(listener);
        }

        listenersByKey.remove(listenerKey);

        // Also remove the listener from the global listeners!
        globalListeners.remove(listener);
    }

    private void sendEventTo(Event event, Collection listeners)
    {
        if (listeners == null || listeners.size() == 0)
            return;

        for (Iterator it = listeners.iterator(); it.hasNext();)
        {
            com.atlassian.event.EventListener eventListener = (com.atlassian.event.EventListener) it.next();
            eventListener.handleEvent(event);
        }
    }

    private void addToListenerList(Class aClass, com.atlassian.event.EventListener listener)
    {
        if (!listenersByClass.containsKey(aClass))
            listenersByClass.put(aClass, new ArrayList());

        ((List)listenersByClass.get(aClass)).add(listener);
    }
}
