package com.atlassian.diagnostics.internal.platform.monitor.thread;

import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.jmx.ThreadMemoryAllocation;
import com.atlassian.diagnostics.internal.platform.plugin.PluginFinder;
import com.google.common.collect.ImmutableMap;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class ThreadMonitor extends InitializingMonitor {

    private static final String KEY_PREFIX = "diagnostics.thread.issue";
    private static final int HIGH_THREAD_MEMORY_USAGE = 2001;

    private final ThreadMonitorConfiguration threadMonitorConfiguration;
    private final PluginFinder pluginFinder;
    private final ThreadDumpProducer threadDumpProducer;

    public ThreadMonitor(final ThreadMonitorConfiguration threadMonitorConfiguration,
                         final PluginFinder pluginFinder,
                         final ThreadDumpProducer threadDumpProducer) {
        this.threadMonitorConfiguration = threadMonitorConfiguration;
        this.pluginFinder = pluginFinder;
        this.threadDumpProducer = threadDumpProducer;
    }

    @Override
    public void init(final MonitoringService monitoringService) {
        monitor = monitoringService.createMonitor("THREAD", "diagnostics.thread.name", threadMonitorConfiguration);

        defineIssue(KEY_PREFIX, HIGH_THREAD_MEMORY_USAGE, Severity.WARNING);
    }

    public void raiseAlertForHighThreadMemoryUsage(final Instant now, final List<ThreadMemoryAllocation> threadMemoryAllocations) {
        alert(HIGH_THREAD_MEMORY_USAGE, builder -> builder
                .timestamp(now)
                .details(() -> highThreadMemoryUsageAlertDetails(threadMemoryAllocations))
        );
    }

    private Map<String, Object> highThreadMemoryUsageAlertDetails(final List<ThreadMemoryAllocation> threadMemoryAllocations) {
        final List<Map<String, Object>> threadsWithHighMemoryUsageAlertDetails = threadMemoryAllocations.stream()
                .map(toAlertDetails())
                .collect(Collectors.toList());

        return new ImmutableMap.Builder<String, Object>()
                .put("threadsWithHighMemoryUsage", threadsWithHighMemoryUsageAlertDetails)
                .build();
    }

    private Function<ThreadMemoryAllocation, Map<String, Object>> toAlertDetails() {
        return threadMemoryAllocation -> {
            final ImmutableMap.Builder<String, Object> builder = new ImmutableMap.Builder<String, Object>()
                    .put("memoryAllocationInBytes", threadMemoryAllocation.getMemoryAllocationInBytes())
                    .put("threadName", threadMemoryAllocation.getThreadName());

            final String stackTrace = threadDumpProducer.toStackTraceString(Arrays.asList(threadMemoryAllocation.getStackTrace()));
            if (isNotEmpty(stackTrace)) {
                builder.put("stackTrace", stackTrace);
            }

            final Collection<String> plugins = pluginFinder.getPluginNamesFromStackTrace(threadMemoryAllocation.getStackTrace());
            if (!plugins.isEmpty()) {
                builder.put("plugins", String.join(" -> ", plugins));
            }

            return builder.build();
        };
    }
}
