package com.atlassian.diagnostics.internal.platform.monitor.db;

import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.platform.plugin.PluginFinder;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Map;

/**
 * Monitor for database related alerts.
 * {@link DatabaseMonitor#init} must be called to begin operations
 */
public class DatabaseMonitor extends InitializingMonitor {

    private static final Logger logger = LoggerFactory.getLogger(DatabaseMonitor.class);
    private static final String KEY_PREFIX = "diagnostics.db.issue";
    private static final int DB_POOL_CONNECTION_LEAK_ID = 2001;
    private static final int DB_SLOW_OPERATION_ISSUE_ID = 3001;
    private static final int DB_POOL_HIGH_UTILIZATION_ISSUE_ID = 3002;

    private final DatabaseMonitorConfiguration databaseMonitorConfiguration;
    private final ThreadDumpProducer threadDumpProducer;
    private final PluginFinder pluginFinder;

    public DatabaseMonitor(@Nonnull final DatabaseMonitorConfiguration databaseMonitorConfiguration,
                           @Nonnull final ThreadDumpProducer threadDumpProducer,
                           @Nonnull final PluginFinder pluginFinder) {
        this.databaseMonitorConfiguration = databaseMonitorConfiguration;
        this.threadDumpProducer = threadDumpProducer;
        this.pluginFinder = pluginFinder;
    }

    @Override
    public void init(final MonitoringService monitoringService) {
        logger.info("Initializing DatabaseMonitor");
        monitor = monitoringService.createMonitor("DB", "diagnostics.db.name", databaseMonitorConfiguration);
        defineIssue(KEY_PREFIX, DB_POOL_CONNECTION_LEAK_ID, Severity.WARNING);
        defineIssue(KEY_PREFIX, DB_SLOW_OPERATION_ISSUE_ID, Severity.INFO);
        defineIssue(KEY_PREFIX, DB_POOL_HIGH_UTILIZATION_ISSUE_ID, Severity.INFO);
    }

    /**
     * Raises an alert for connection leak.
     * @param timestamp the timestamp to be used on the alert
     * @param connectionAcquiredTimestamp the connection acquisition timestamp
     * @param diagnostic the diagnostic details
     */
    public void raiseAlertForConnectionLeak(@Nonnull final Instant timestamp,
                                            @Nonnull final Instant connectionAcquiredTimestamp,
                                            @Nonnull final DatabasePoolDiagnostic diagnostic) {
        alert(DB_POOL_CONNECTION_LEAK_ID, builder -> builder
                .timestamp(timestamp)
                .details(() -> connectionLeakAlertDetails(diagnostic, connectionAcquiredTimestamp))
        );
    }

    private Map<Object, Object> connectionLeakAlertDetails(final DatabasePoolDiagnostic diagnostic, final Instant connectionAcquiredTimestamp) {
        return ImmutableMap.builder()
                .put("activeConnections", diagnostic.getActiveConnections())
                .put("idleConnections", diagnostic.getIdleConnections())
                .put("maxConnections", diagnostic.getMaxConnections())
                .put("connectionAcquiredTimestampInMillis", connectionAcquiredTimestamp.toEpochMilli())
                .build();
    }

    /**
     * Raises an alert for high connection pool utilization.
     * @param timestamp the timestamp to be used on the alert
     * @param diagnostic the snapshot of the database pool
     */
    public void raiseAlertForHighPoolUtilization(@Nonnull final Instant timestamp, @Nonnull final DatabasePoolDiagnostic diagnostic) {
        alert(DB_POOL_HIGH_UTILIZATION_ISSUE_ID, builder -> builder
                .timestamp(timestamp)
                .details(() -> highUtilizationAlertDetails(diagnostic))
        );
    }

    private Map<Object, Object> highUtilizationAlertDetails(final DatabasePoolDiagnostic diagnostic) {
        return ImmutableMap.builder()
                .put("activeConnections", diagnostic.getActiveConnections())
                .put("idleConnections", diagnostic.getIdleConnections())
                .put("maxConnections", diagnostic.getMaxConnections())
                .build();
    }

    /**
     * Raises an alert for slow running database operations.
     * @param timestamp the timestamp to be used on the alert
     * @param diagnostic the slow running query
     */
    public void raiseAlertForSlowOperation(@Nonnull final Instant timestamp, @Nonnull final DatabaseOperationDiagnostic diagnostic) {
        alert(DB_SLOW_OPERATION_ISSUE_ID, builder -> builder
                .timestamp(timestamp)
                .details(() -> slowOperationAlertDetails(diagnostic))
        );
    }

    private Map<String, Object> slowOperationAlertDetails(final DatabaseOperationDiagnostic diagnostic) {
        final ImmutableMap.Builder<String, Object> map = ImmutableMap.<String, Object>builder()
                .put("executionTimeInMillis", diagnostic.getExecutionTime().toMillis())
                .put("theadDump", threadDumpProducer.produce(ImmutableSet.of(Thread.currentThread())))
                .put("plugins", String.join(" -> ", pluginFinder.getPluginNamesInCurrentCallStack()));

        if (databaseMonitorConfiguration.includeSqlQueryInAlerts()) {
            map.put("sql", diagnostic.getSql());
        }

        return map.build();
    }

}
