package com.atlassian.diagnostics.internal.platform.monitor.http;

import com.atlassian.diagnostics.internal.AlertProvider;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.user.UserProfile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;

public class HttpRequestMonitoringFilter extends AlertProvider<HttpMonitorConfiguration> implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(HttpRequestMonitoringFilter.class);

    private final HttpRequestMonitor monitor;
    private final UserManager userManager;
    private final Clock clock;
    private final I18nResolver i18nResolver;
    private final String key = HttpRequestMonitor.class.getName();
    private final String I18N_NO_USERNAME_DETECTED = "diagnostics.http.no.username";

    public HttpRequestMonitoringFilter(@Nonnull final HttpRequestMonitor monitor,
                                       @Nonnull final UserManager userManager,
                                       @Nonnull final HttpMonitorConfiguration config,
                                       @Nonnull final Clock clock,
                                       @Nonnull final I18nResolver i18nResolver) {
        super(HttpRequestMonitor.class.getName(), config);
        this.monitor = monitor;
        this.userManager = userManager;
        this.clock = clock;
        this.i18nResolver = i18nResolver;
    }

    @Override
    public void init(final FilterConfig filterConfig) {}

    @Override
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
        final Instant requestStartTime = clock.instant();
        final String username = getUsernameFromRequest(request);
        try {
            chain.doFilter(request, response);
        } finally {
            try {
                final Instant requestEndTime = clock.instant();
                if (isMonitoringEnabledFor(request) && monitorConfiguration.isEnabled() && isRequestSlow(requestStartTime, requestEndTime)) {
                    raiseAlertForSlowHttpRequest(requestStartTime, requestEndTime, (HttpServletRequest) request, username);
                }
            } catch (final Exception e) {
                logger.debug("Failed to raise alert", e);
            }
        }
    }

    private boolean isMonitoringEnabledFor(final ServletRequest request) {
        if (request instanceof HttpServletRequest) {
            final String path = ((HttpServletRequest) request).getRequestURI();
            return path.contains("/rest/") || path.contains("/servlet/");
        }
        return false;
    }

    private boolean isRequestSlow(final Instant requestStartTime, final Instant requestFinishTime) {
        return Duration.between(requestStartTime, requestFinishTime).toMillis() > monitorConfiguration.getMaximumHttpRequestTime().toMillis();
    }

    private void raiseAlertForSlowHttpRequest(final Instant requestStartTime, final Instant requestEndTime, final HttpServletRequest request, final String username) {
        final String url = (request != null && request.getRequestURI() != null) ? request.getRequestURI() : "";
        final HttpRequestDiagnostic diagnostic = new HttpRequestDiagnostic(url, username, Duration.between(requestStartTime, requestEndTime));

        monitor.raiseAlertForSlowHttpRequest(Instant.now(), diagnostic);
    }

    private String getUsernameFromRequest(final ServletRequest httpRequest) {
        HttpServletRequest httpServletRequest = (HttpServletRequest) httpRequest;
        if (httpServletRequest.getRemoteUser() == null) {
            return i18nResolver.getText(I18N_NO_USERNAME_DETECTED);
        }
        final Optional<UserProfile> remoteUser = Optional.ofNullable(userManager.getRemoteUser(httpServletRequest));
        return remoteUser.isPresent() ? remoteUser.get().getUsername() : i18nResolver.getText(I18N_NO_USERNAME_DETECTED);
    }

    @Override
    public void destroy() {}

    @Override
    public String getKey() {
        return key;
    }
}
