package com.atlassian.diagnostics.internal.platform.monitor.gc;

import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;
import java.time.Clock;
import java.time.Instant;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

@NotThreadSafe
@ParametersAreNonnullByDefault
public class GarbageCollectionPoller extends DiagnosticPoller<GarbageCollectionMonitorConfiguration> {

    private final GarbageCollectionMonitorConfiguration garbageCollectionMonitorConfiguration;
    private final GCMXBeanPoller gcMXBeanPoller;
    private final Clock clock;
    private final GCReadsStore gcReadsStore;
    private final GarbageCollectionMonitor garbageCollectionMonitor;

    public GarbageCollectionPoller(final GarbageCollectionMonitor garbageCollectionMonitor,
                                   final GarbageCollectionMonitorConfiguration garbageCollectionMonitorConfiguration,
                                   final GCMXBeanPollerFactory gcMXBeanPollerFactory,
                                   final GCReadsStore gcReadsStore,
                                   final Clock clock) {
        super(GarbageCollectionPoller.class.getName(), garbageCollectionMonitorConfiguration);
        this.garbageCollectionMonitor = requireNonNull(garbageCollectionMonitor);
        this.garbageCollectionMonitorConfiguration = requireNonNull(garbageCollectionMonitorConfiguration);
        this.clock = requireNonNull(clock);
        this.gcReadsStore = requireNonNull(gcReadsStore);

        gcMXBeanPoller = gcMXBeanPollerFactory
                .getGCMXBeansPoller()
                .orElse(null);
    }

    @Override
    protected void execute() {
        if (gcMXBeanPollerPresent()) {
            raiseAlertIfGarbageCollectionTimeIsAboveThreshold();
        }
    }

    private boolean gcMXBeanPollerPresent() {
        return gcMXBeanPoller != null;
    }

    private void raiseAlertIfGarbageCollectionTimeIsAboveThreshold() {
        final GCRead currentGCRead = gcMXBeanPoller.doRead();
        gcReadsStore.storeRead(currentGCRead);

        final Optional<GCRead> readFromBeforeTheTimeWindow = gcReadsStore.getReadIfHappenedBefore(timeWindowStart());
        readFromBeforeTheTimeWindow.ifPresent(previousRead -> raiseAlertIfGarbageCollectionTimeIsAboveThreshold(previousRead, currentGCRead));
    }

    private Instant timeWindowStart() {
        return Instant.now(clock).minus(garbageCollectionMonitorConfiguration.slidingWindowSize());
    }

    private void raiseAlertIfGarbageCollectionTimeIsAboveThreshold(final GCRead previousRead, final GCRead currentRead) {
        final GCDetailsCalculator alertDetailsCalculator = new GCDetailsCalculator(previousRead, currentRead);

        if (shouldRaiseErrorAlert(alertDetailsCalculator)) {
            garbageCollectionMonitor.raiseAlertForHighGarbageCollectionTime(
                    getDetailsBuilder(alertDetailsCalculator)
                            .severity(Severity.ERROR)
                            .build());
        } else if (shouldRaiseWarningAlert(alertDetailsCalculator)) {
            garbageCollectionMonitor.raiseAlertForHighGarbageCollectionTime(
                    getDetailsBuilder(alertDetailsCalculator)
                            .severity(Severity.WARNING)
                            .build());
        }
    }

    public boolean shouldRaiseErrorAlert(GCDetailsCalculator alertDetailsCalculator) {
        return garbageCollectionTimeExceedsThreshold(alertDetailsCalculator, monitorConfiguration.getErrorThreshold());
    }

    public boolean shouldRaiseWarningAlert(GCDetailsCalculator alertDetailsCalculator) {
        return garbageCollectionTimeExceedsThreshold(alertDetailsCalculator, monitorConfiguration.getWarningThreshold());
    }

    private boolean garbageCollectionTimeExceedsThreshold(GCDetailsCalculator alertDetailsCalculator, final double percentageThreshold) {
        return alertDetailsCalculator.getPercentageOfTimeInGarbageCollection() >= percentageThreshold;
    }

    private HighGCTimeDetails.HighGCTimeAlertBuilder getDetailsBuilder(GCDetailsCalculator alertCalculator) {
        return HighGCTimeDetails.builder()
                .timestamp(Instant.now(clock))
                .addAlertInfo(alertCalculator)
                .garbageCollectorName(gcMXBeanPoller.getGarbageCollectorName());
    }
}
