package com.atlassian.diagnostics.internal.platform.plugin;

import com.atlassian.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.atlassian.fugue.Pair;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class PluginFinderImpl implements PluginFinder {
    private static final Logger logger = LoggerFactory.getLogger(PluginFinderImpl.class);

    private final ClassContextSecurityManager securityManger;
    private final BundleContext bundleContext;
    private final BundleFinder bundleFinder;
    private final Cache<Class<?>, String> classPluginSourceCache;
    private final Cache<String, Class<?>> classNameClassSourceCache;

    public PluginFinderImpl(final BundleContext bundleContext, final BundleFinder bundleFinder) {
        this(createClassContextSecurityManager(),
                bundleContext,
                bundleFinder,
                CacheBuilder.newBuilder()
                        .maximumSize(10_000)
                        .weakKeys()
                        .expireAfterAccess(Duration.ofHours(1))
                        .build(),
                CacheBuilder.newBuilder()
                        .maximumSize(10_000)
                        .weakValues()
                        .expireAfterAccess(Duration.ofHours(1))
                        .build()
        );
    }

    private static ClassContextSecurityManager createClassContextSecurityManager() {
        try {
            return new ClassContextSecurityManager();
        } catch (Exception e) {
            logger.debug("Failed to create security manager", e);
            return null;
        }
    }

    @VisibleForTesting
    PluginFinderImpl(final ClassContextSecurityManager securityManger,
                     final BundleContext bundleContext,
                     final BundleFinder bundleFinder,
                     final Cache<Class<?>, String> classPluginSourceCache,
                     final Cache<String, Class<?>> classNameClassSourceCache) {
        this.securityManger = securityManger;
        this.bundleContext = bundleContext;
        this.bundleFinder = bundleFinder;
        this.classPluginSourceCache = classPluginSourceCache;
        this.classNameClassSourceCache = classNameClassSourceCache;
    }

    static class ClassContextSecurityManager extends SecurityManager {
        private static final Class<?>[] EMPTY_ARRAY = new Class<?>[0];

        @Override
        protected Class<?>[] getClassContext() {
            final Class<?>[] classContext = super.getClassContext();
            return (classContext == null)
                    ? EMPTY_ARRAY
                    : classContext;
        }
    }

    public Collection<String> getPluginNamesInCurrentCallStack() {
        try {
            if (securityManger != null) {
                return getPluginsFromClasses(securityManger.getClassContext());
            }
        } catch (Exception e) {
            logger.debug("Failed to get plugins list from call stack", e);
        }
        return Collections.emptyList();
    }

    @Override
    public Collection<String> getPluginNamesFromStackTrace(@Nonnull final StackTraceElement[] stackTrace) {
        return getPluginsFromClasses(resolveClassesFromStackTrace(stackTrace));
    }

    private Class<?>[] resolveClassesFromStackTrace(@Nonnull final StackTraceElement[] stackTrace) {
        final Set<Class<?>> classes = new HashSet<>();
        for (final StackTraceElement stackTraceElement : stackTrace) {
            final Class<?> cachedClass = classNameClassSourceCache.getIfPresent(stackTraceElement.getClassName());
            if (cachedClass != null) {
                classes.add(cachedClass);
            } else {
                final Class<?> classFromName = resolveClass(stackTraceElement.getClassName());
                if (classFromName != null) {
                    classNameClassSourceCache.put(stackTraceElement.getClassName(), classFromName);
                    classes.add(classFromName);
                }
            }
        }

        return classes.toArray(new Class<?>[0]);
    }

    private Class<?> resolveClass(final String className) {
        try {
            return Class.forName(className);
        } catch (ClassNotFoundException | LinkageError e) {
            final Map<Bundle, Class<?>> bundlesWithClass = Arrays.stream(bundleContext.getBundles())
                    .filter(bundle -> bundle.getState() == Bundle.ACTIVE)
                    .filter(bundle -> bundle.getHeaders().get("Atlassian-Plugin-Key") != null)
                    .map(bundle -> {
                        try {
                            return new Pair<Bundle, Class<?>>(bundle, bundle.loadClass(className));
                        } catch (ClassNotFoundException | LinkageError ex) {
                            return null;
                        }
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.toMap(Pair::left, Pair::right));
            if (bundlesWithClass.size() == 1) {
                return bundlesWithClass.entrySet().iterator().next().getValue();
            } else if (bundlesWithClass.size() > 1) {
                logger.debug("Couldn't resolve class name [{}] as it was found in {} bundles: {}", className, bundlesWithClass.size(), bundlesWithClass.keySet());
            }

            return null;
        }
    }

    private Collection<String> getPluginsFromClasses(final Class<?>[] classes) {
        final Set<String> plugins = new HashSet<>();
        for (final Class<?> clazz : classes) {
            final String cachedPluginSource = classPluginSourceCache.getIfPresent(clazz);

            if (cachedPluginSource == null) {
                resolvePlugin(clazz).ifPresent(pluginName -> add(plugins, pluginName));
            } else if (isNotEmpty(cachedPluginSource)) {
                add(plugins, cachedPluginSource);
            }
        }

        return plugins;
    }

    private Optional<String> resolvePlugin(final Class<?> clazz) {
        final Optional<String> pluginName = bundleFinder.getBundleNameForClass(clazz);
        classPluginSourceCache.put(clazz, pluginName.orElse(""));

        return pluginName;
    }

    private void add(final Set<String> plugins, final String pluginName) {
        plugins.add(pluginName);
    }
}
