package com.atlassian.diagnostics.internal.platform.monitor.db;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;

import static com.atlassian.diagnostics.internal.platform.monitor.DurationUtils.durationOf;

public class DefaultDatabaseDiagnosticsCollector implements DatabaseDiagnosticsCollector {
    private static final Logger logger = LoggerFactory.getLogger(DefaultDatabaseDiagnosticsCollector.class);

    private final DatabaseMonitorConfiguration configuration;
    private final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider;
    private final Duration poolConnectionLeakTimeout;
    private final DatabaseMonitor databaseMonitor;
    private final Clock clock;
    private final Cache<Connection, Instant> connectionCache;

    public DefaultDatabaseDiagnosticsCollector(@Nonnull final DatabaseMonitorConfiguration configuration,
                                               @Nonnull final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider,
                                               @Nonnull final Clock clock,
                                               @Nonnull final DatabaseMonitor databaseMonitor) {
        this.configuration = configuration;
        this.databasePoolDiagnosticProvider = databasePoolDiagnosticProvider;
        this.databaseMonitor = databaseMonitor;
        this.poolConnectionLeakTimeout = configuration.poolConnectionLeakTimeout();
        this.clock = clock;
        this.connectionCache = CacheBuilder.newBuilder()
                .weakKeys()
                .maximumSize(500)
                .expireAfterAccess(poolConnectionLeakTimeout)
                .removalListener(new LeakedConnectionListener())
                .build();
    }

    @Override
    public boolean isEnabled() {
        return configuration.isEnabled();
    }

    @Override
    public void trackConnection(final Connection connection) {
        if (durationOf(poolConnectionLeakTimeout).isGreaterThan(Duration.ZERO)) {
            connectionCache.put(connection, clock.instant());
        }
    }

    @Override
    public void removeTrackedConnection(final Connection connection) {
        if (durationOf(poolConnectionLeakTimeout).isGreaterThan(Duration.ZERO)) {
            connectionCache.invalidate(connection);
        }
    }

    @Override
    public <T> T recordExecutionTime(final SqlOperation<T> operation, final String sql) throws SQLException {
        final long startTime = System.currentTimeMillis();
        try {
            return operation.execute();
        } finally {
            try {
                final Duration duration = Duration.ofMillis(System.currentTimeMillis() - startTime);
                raiseAlertIfExecutionExceededThreshold(sql, duration);
            } catch (Exception e) {
                logger.debug("Failed to raise alert", e);
            }
        }
    }

    private void raiseAlertIfExecutionExceededThreshold(final String sql, final Duration duration) {
        if (durationOf(duration).isGreaterThanOrEqualTo(configuration.longRunningOperationLimit())) {
            final DatabaseOperationDiagnostic diagnostic = new DatabaseOperationDiagnostic(sql, duration, Thread.currentThread().getName());
            databaseMonitor.raiseAlertForSlowOperation(Instant.now(), diagnostic);
        }
    }

    private class LeakedConnectionListener implements RemovalListener<Connection, Instant> {

        @Override
        public void onRemoval(final RemovalNotification<Connection, Instant> notification) {
            if (notification.getCause() == RemovalCause.EXPIRED) {
                final DatabasePoolDiagnostic databasePoolDiagnostic = databasePoolDiagnosticProvider.getDiagnostic();
                if (!databasePoolDiagnostic.isEmpty()) {
                    databaseMonitor.raiseAlertForConnectionLeak(clock.instant(), notification.getValue(), databasePoolDiagnostic);
                }
            }
        }
    }

}
