package com.atlassian.diagnostics.internal.platform.event;

import com.atlassian.event.spi.EventDispatcher;
import com.atlassian.event.spi.ListenerInvoker;

import java.time.Instant;
import java.util.concurrent.RejectedExecutionException;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Wrapper around an {@link EventDispatcher} that raises an alert when an event is rejected.
 */
public class MonitoredEventDispatcher implements EventDispatcher {

    private final EventDispatcher delegate;
    private final EventSystemMonitor monitor;
    private final Supplier<Integer> queueLengthSupplier;

    public MonitoredEventDispatcher(EventDispatcher delegate, EventSystemMonitor monitor,
                                    int queueLength) {
        this(delegate, monitor, () -> queueLength);
    }

    public MonitoredEventDispatcher(EventDispatcher delegate, EventSystemMonitor monitor,
                                    Supplier<Integer> queueLengthSupplier) {
        this.delegate = requireNonNull(delegate, "delegate");
        this.monitor = requireNonNull(monitor, "monitor");
        this.queueLengthSupplier = requireNonNull(queueLengthSupplier, "queueLengthSupplier");
    }

    @Override
    public void dispatch(ListenerInvoker listenerInvoker, Object o) {
        try {
            delegate.dispatch(listenerInvoker, o);
        } catch (RejectedExecutionException e) {
            monitor.alertEventDropped(Instant.now(), queueLengthSupplier.get(), o.getClass());
            throw e;
        }
    }
}
