package com.atlassian.diagnostics.internal.platform.event;

import com.atlassian.diagnostics.AlertRequest;
import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.ComponentMonitor;
import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.JacksonJsonMapper;
import com.atlassian.diagnostics.internal.concurrent.Gate;
import com.atlassian.event.spi.ListenerInvoker;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

/**
 * Monitor for the event system that encapsulates all interactions with the atlassian-diagnostics framework
 */
public class EventSystemMonitor implements InitializingMonitor {

    static final int ID_EVENT_DROPPED = 1001;
    static final int ID_SLOW_LISTENER = 2001;

    private final EventSystemMonitorConfig config;
    private final Object lock;
    private final Set<Thread> invokerThreads;
    private final ThreadDumpProducer threadDumpProducer;
    private final Gate threadDumpGate;

    private Clock clock;
    private volatile ComponentMonitor monitor;

    EventSystemMonitor(@Nonnull Clock clock, @Nonnull EventSystemMonitorConfig config,
                       @Nonnull ThreadDumpProducer threadDumpProducer) {
        this.clock = requireNonNull(clock, "clock");
        this.config = requireNonNull(config, "config");
        this.threadDumpGate = new Gate(clock, config.getEventDroppedAlertThreadDumpCoolDown());
        this.threadDumpProducer = requireNonNull(threadDumpProducer, "threadDumpProducer");

        lock = new Object();
        invokerThreads = Sets.newConcurrentHashSet();
    }

    public EventSystemMonitor(@Nonnull EventSystemMonitorConfig config,
                              @Nonnull ThreadDumpProducer threadDumpProducer) {
        this(Clock.systemDefaultZone(), config, threadDumpProducer);
    }

    @Override
    public void init(MonitoringService monitoringService) {
        synchronized (lock) {
            ComponentMonitor mon = monitoringService.createMonitor("EVENT", "diagnostics.event.name");

            defineIssues(mon);
            monitor = mon;
        }
    }

    protected void defineIssues(ComponentMonitor monitor) {
        defineIssue(monitor, ID_EVENT_DROPPED, Severity.ERROR, EventDroppedDetails.class);
        defineIssue(monitor, ID_SLOW_LISTENER, Severity.WARNING, null);
    }

    /**
     * Raises an alert that an event has been dropped
     *  @param timestamp the timestamp to be used on the alert
     * @param queueLength the length of the (full) event queue
     * @param eventClass the type of event that was dropped
     */
    void alertEventDropped(@Nonnull Instant timestamp, int queueLength, @Nonnull Class<?> eventClass) {
        requireNonNull(timestamp, "timestamp");

        alert(ID_EVENT_DROPPED, builder -> builder
                .timestamp(timestamp)
                .details(() -> {
                    EventDroppedDetails.Builder detailsBuilder =
                            new EventDroppedDetails.Builder(eventClass.getName(), queueLength);

                    threadDumpGate.ifAccessible(() -> {
                        detailsBuilder.threadDumps(threadDumpProducer.produce(getEventPoolThreads()));
                        return null;
                    });
                    return detailsBuilder.build();
                }));
    }

    /**
     * Invokes the listener and monitors the invocation
     *
     * @param trigger the trigger for the event
     * @param delegate the delete for invocation
     * @param event the event
     */
    void invokeMonitored(@Nonnull AlertTrigger trigger, @Nonnull ListenerInvoker delegate, @Nonnull Object event) {
        Instant start = clock.instant();
        invokerThreads.add(Thread.currentThread());
        try {
            delegate.invoke(event);
        } finally {
            invokerThreads.remove(Thread.currentThread());
            Duration duration = Duration.between(start, clock.instant());
            if (duration.compareTo(config.getSlowListenerAlertDuration(trigger)) >= 0) {
                alert(ID_SLOW_LISTENER, builder -> builder
                        .timestamp(start)
                        .trigger(trigger)
                        .details(() -> ImmutableMap.of(
                                "timeMillis", duration.toMillis(),
                                "eventType", event.getClass().getName())));
            }
        }
    }

    private static void defineIssue(ComponentMonitor monitor, int id, Severity severity, Class<?> detailsClass) {
        String i18nPrefix = "diagnostics.event.issue." + StringUtils.leftPad(Integer.toString(id), 4, '0') + ".";
        monitor.defineIssue(id)
                .summaryI18nKey(i18nPrefix + "summary")
                .descriptionI18nKey(i18nPrefix + "description")
                .jsonMapper(detailsClass == null ? null : new JacksonJsonMapper<>(detailsClass))
                .severity(severity)
                .build();
    }

    private void alert(int issueId, Consumer<AlertRequest.Builder> alertBuilder) {
        ComponentMonitor mon = monitor;
        if (mon != null && mon.isEnabled()) {
            mon.getIssue(issueId).ifPresent(issue -> {
                AlertRequest.Builder builder = new AlertRequest.Builder(issue);
                alertBuilder.accept(builder);
                mon.alert(builder.build());
            });
        }
    }

    private Set<Thread> getEventPoolThreads() {
        // only return threads that are part of the event dispatcher thread pool
        return config.getEventThreadGroup()
                .map(threadGroup ->
                        invokerThreads.stream()
                        .filter(thread -> thread.getThreadGroup().equals(threadGroup))
                        .collect(Collectors.toSet()))
                .orElse(invokerThreads);
    }
}
