package com.atlassian.diagnostics.internal.platform.event;

import com.atlassian.diagnostics.detail.ThreadDump;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Details object that's produced in response to {@link EventSystemMonitor#ID_EVENT_DROPPED 'event dropped'} alerts.
 */
public class EventDroppedDetails {

    private final String eventType;
    private final int queueLength;
    private final List<ThreadDump> threadDumps;

    private EventDroppedDetails(Builder builder) {
        this.eventType = builder.eventType;
        this.queueLength = builder.queueLength;
        this.threadDumps = builder.threadDumps.build();
    }

    @JsonCreator
    static EventDroppedDetails create(@JsonProperty("eventType") String className,
                                      @JsonProperty("queueLength") int queueLength,
                                      @JsonProperty("threads") List<ThreadDump> threadDumps) {
        Builder builder = new Builder(className, queueLength);
        if (threadDumps != null) {
            builder.threadDumps(threadDumps);
        }
        return builder.build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        EventDroppedDetails details = (EventDroppedDetails) o;
        return queueLength == details.queueLength &&
                Objects.equals(eventType, details.eventType) &&
                Objects.equals(threadDumps, details.threadDumps);
    }

    @Nonnull
    public String getEventType() {
        return eventType;
    }

    public int getQueueLength() {
        return queueLength;
    }

    @Nonnull
    public List<ThreadDump> getThreadDumps() {
        return threadDumps;
    }

    @Override
    public int hashCode() {
        return Objects.hash(eventType, queueLength, threadDumps);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("eventType", eventType)
                .add("queueLength", queueLength)
                .add("threadDumps", threadDumps)
                .toString();
    }

    public static class Builder {

        private final String eventType;
        private final int queueLength;
        private final ImmutableList.Builder<ThreadDump> threadDumps;

        public Builder(String eventClass, int queueLength) {
            this.eventType = requireNonNull(eventClass, "eventType");
            this.queueLength = queueLength;

            threadDumps = ImmutableList.builder();
        }

        @Nonnull
        public EventDroppedDetails build() {
            return new EventDroppedDetails(this);
        }

        @Nonnull
        public Builder threadDumps(@Nonnull Iterable<ThreadDump> values) {
            threadDumps.addAll(requireNonNull(values, "threadDumps"));
            return this;
        }
    }
}
